/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _ACES_ODT_
#define _ACES_ODT_

/* Global ODT Settings */

#define ODT_CINEMA_WHITE 48.0
const float ODT_CINEMA_BLACK = ODT_CINEMA_WHITE / 2400.0;

#define DISP_GAMMA 2.4
#define OFFSET 0.055

#define DIM_SURROUND_GAMMA 0.9811

#define ODT_SAT_FACTOR 0.93

#define L_W 1.0
#define L_B 0.0

/* Display Primaries Matrices */

const mat3 XYZ_2_REC709_PRI_MAT = mat3(
     3.2409699419, -1.5373831776, -0.4986107603,
    -0.9692436363,  1.8759675015,  0.0415550574,
     0.0556300797, -0.2039769589,  1.0569715142
);

const mat3 XYZ_2_P3D60_PRI_MAT = mat3(
     2.4027414142, -0.8974841639, -0.3880533700,
    -0.8325796487,  1.7692317536,  0.0237127115,
     0.0388233815, -0.0824996856,  1.0363685997
);

/* ODT Prerequisite Function */
// This code is common to all ODT's Continuum offers, so it's in its own function to make organisation easier.

vec3 Y_2_linCV(vec3 Y, const float Ymax, const float Ymin) {
    return (Y - Ymin) / (Ymax - Ymin);
}

vec3 ODT_pre(vec3 color_OCES) {
    vec3 color_RGBpre = color_OCES * AP0_2_AP1;

    SegmentedSplineParams_c9 odtParams = ODT_48nits;

    odtParams.minPoint.x = segmented_spline_c5_fwd(odtParams.minPoint.x, RRT_PARAMS);
    odtParams.midPoint.x = segmented_spline_c5_fwd(odtParams.midPoint.x, RRT_PARAMS);
    odtParams.maxPoint.x = segmented_spline_c5_fwd(odtParams.maxPoint.x, RRT_PARAMS);

    // Apply tonescale independently in RGB
    vec3 color_RGBpost;
    color_RGBpost.r = segmented_spline_c9_fwd(color_RGBpre.r, odtParams);
    color_RGBpost.g = segmented_spline_c9_fwd(color_RGBpre.g, odtParams);
    color_RGBpost.b = segmented_spline_c9_fwd(color_RGBpre.b, odtParams);

    // Scale luminance to linear code value
    return Y_2_linCV(color_RGBpost, ODT_CINEMA_WHITE, ODT_CINEMA_BLACK);
}

/* Dark -> Dim Surround Gamma Adjustment */

vec3 darkSurround_to_dimSurround(vec3 linearCV) {
    vec3 XYZ = linearCV * AP1_2_XYZ;

    vec3 xyY = XYZ_2_xyY(XYZ);
    xyY.z = max0(xyY.z);
    xyY.z = pow(xyY.z, DIM_SURROUND_GAMMA);
    XYZ = xyY_2_XYZ(xyY);

    return XYZ * XYZ_2_AP1;
}

/* sRGB 100nits Dim ODT */

float moncurve_r(float y, const float gamma, const float offs) {
    const float yb = pow(offs * gamma / ((gamma - 1.0) * (1.0 + offs)), gamma);
    const float rs = pow((gamma - 1.0) / offs, gamma - 1.0) * pow((1.0 + offs) / gamma, gamma);
    return y >= yb ? (1.0 + offs) * pow(y, 1.0 / gamma) - offs : y * rs;
}

vec3 ODT_sRGB_100nits_dim(vec3 color_RGB) {
    // Apply gamma adjustment to compensate for dim surround
    color_RGB = darkSurround_to_dimSurround(color_RGB);

    // Apply desat to compensate for luminance difference
    color_RGB = color_RGB * calc_sat_adjust_matrix(ODT_SAT_FACTOR, AP1_RGB2Y);

    // Convert to display primary encoding
    // RGB -> XYZ
    vec3 color_XYZ = color_RGB * AP1_2_XYZ;

    // Apply CAT from ACES white point to assumed observer adapted white point
    color_XYZ = color_XYZ * D60_2_D65_CAT;

    // CIE XYZ -> Display Primaries
    color_RGB = color_XYZ * XYZ_2_REC709_PRI_MAT;

    // Handle out-of-gamut values
    // Clip values <0 or >1 (ie projecting outside the display primaries)
    color_RGB = clamp01(color_RGB);

    // Encode linear code values with transfer function
    vec3 outputCV;
    outputCV.r = moncurve_r(color_RGB.r, DISP_GAMMA, OFFSET);
    outputCV.g = moncurve_r(color_RGB.g, DISP_GAMMA, OFFSET);
    outputCV.b = moncurve_r(color_RGB.b, DISP_GAMMA, OFFSET);

    return outputCV;
}

/* Rec709 100nits Dim ODT */

vec3 bt1886_r(vec3 L, const float gamma, const float Lw, const float Lb) {
    float rGamma = 1.0 / gamma;
    float a = pow(pow(Lw, rGamma) - pow(Lb, rGamma), gamma);
    float b = pow(Lb, rGamma) / (pow(Lw, rGamma) - pow(Lb, rGamma));
    return pow(max(L / a, 0.0), vec3(rGamma)) - b;
}

vec3 ODT_Rec709_100nits_dim(vec3 color_RGB) {
    // Apply gamma adjustment to compensate for dim surround
    color_RGB = darkSurround_to_dimSurround(color_RGB);

    // Apply desat to compensate for luminance difference
    color_RGB = color_RGB * calc_sat_adjust_matrix(ODT_SAT_FACTOR, AP1_RGB2Y);

    // Convert to display primary encoding
    // RGB -> XYZ
    vec3 color_XYZ = color_RGB * AP1_2_XYZ;

    // Apply CAT from ACES white point to assumed observer adapted white point
    color_XYZ = color_XYZ * D60_2_D65_CAT;

    // CIE XYZ -> Display Primaries
    color_RGB = color_XYZ * XYZ_2_REC709_PRI_MAT;

    // Handle out-of-gamut values
    // Clip values <0 or >1 (ie projecting outside the display primaries)
    color_RGB = clamp01(color_RGB);
    
    // Encode linear code values with transfer function
    return bt1886_r(color_RGB, DISP_GAMMA, L_W, L_B);
}

#endif
