/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _ACES_UTILITY_
#define _ACES_UTILITY_

/* Color Space Conversion Matrices */

const mat3 AP0_2_XYZ = mat3(
    0.9525523959, 0.0000000000,  0.0000936786,
    0.3439664498, 0.7281660966, -0.0721325464,
    0.0000000000, 0.0000000000,  1.0088251844
);

const mat3 XYZ_2_AP0 = mat3(
     1.0498110175, 0.0000000000, -0.0000974845,
    -0.4959030231, 1.3733130458,  0.0982400361,
     0.0000000000, 0.0000000000,  0.9912520182
);

const mat3 AP1_2_XYZ = mat3(
     0.6624541811, 0.1340042065, 0.1561876870,
     0.2722287168, 0.6740817658, 0.0536895174,
    -0.0055746495, 0.0040607335, 1.0103391003
);

const mat3 XYZ_2_AP1 = mat3(
     1.6410233797, -0.3248032942, -0.2364246952,
    -0.6636628587,  1.6153315917,  0.0167563477,
     0.0117218943, -0.0082844420,  0.9883948585
);

const mat3 AP0_2_AP1 = mat3(
     1.4514393161, -0.2365107469, -0.2149285693,
    -0.0765537734,  1.1762296998, -0.0996759264,
     0.0083161484, -0.0060324498,  0.9977163014
);

const mat3 AP1_2_AP0 = mat3(
     0.6954522414, 0.1406786965, 0.1638690622,
     0.0447945634, 0.8596711185, 0.0955343182,
    -0.0055258826, 0.0040252103, 1.0015006723
);

const mat3 sRGB_2_XYZ = mat3(
    0.41239089, 0.35758430, 0.18048084,
    0.21263906, 0.71516860, 0.07219233,
    0.01933082, 0.11919472, 0.95053232
);

const mat3 sRGB_2_ACES = mat3(
    0.43963298, 0.38298870, 0.17737832,
    0.08977644, 0.81343943, 0.09678413,
    0.01754117, 0.11154655, 0.87091228
);

const mat3 XYZ_2_sRGB = mat3(
     3.24096942, -1.53738296, -0.49861076,
    -0.96924388,  1.87596786,  0.04155510,
     0.05563002, -0.20397684,  1.05697131
);

const mat3 D65_2_D60_CAT = mat3(
     1.01303000, 0.00610531, -0.01497100,
     0.00769823, 0.99816500, -0.00503203,
    -0.00284131, 0.00468516,  0.92450700
);

const mat3 D60_2_D65_CAT = mat3(
     0.98722400, -0.00611327, 0.01595330,
    -0.00759836,  1.00186000, 0.00533002,
     0.00307257, -0.00509595, 1.08168000
);

const vec3 AP1_RGB2Y = vec3(AP1_2_XYZ[0][1], AP1_2_XYZ[1][1], AP1_2_XYZ[2][1]);

/* Color Space Conversion & Utility Functions */

float rgb_2_saturation(vec3 rgb) {
    const float TINY = 1e-10;
    return (max(TINY, max(max(rgb.r, rgb.g), rgb.b)) - max(min(min(rgb.r, rgb.g), rgb.b), TINY)) / max(max(max(rgb.r, rgb.g), rgb.b), 1e-2);
}

float rgb_2_yc(vec3 rgb, float ycRadiusWeight) { // ycRadiusWeight should default to 1.75
    float chroma = sqrt(rgb.b * (rgb.b - rgb.g) + rgb.g * (rgb.g - rgb.r) + rgb.r * (rgb.r - rgb.b));
    return (rgb.b + rgb.g + rgb.r + ycRadiusWeight * chroma) / 3.0;
}

float rgb_2_hue(vec3 rgb) {
    float hue;
    if(rgb.r == rgb.g && rgb.g == rgb.b) {
        return 0.0; // RGB triplets where RGB are equal have an undefined hue, so return 0
    } else {
        hue = (180.0 * rPI) * atan(sqrt(3) * (rgb.g - rgb.b), 2.0 * rgb.r - rgb.g - rgb.b);
    }
    if(hue < 0.0) hue += 360.0;
    return hue;
}

vec3 XYZ_2_xyY(vec3 XYZ) {
    vec3 xyY;
    float divisor = (XYZ.x + XYZ.y + XYZ.z);
    if (divisor == 0.) divisor = 1e-10;
    xyY.x = XYZ.x / divisor;
    xyY.y = XYZ.y / divisor;
    xyY.z = XYZ.y;
    return xyY;
}

vec3 xyY_2_XYZ(vec3 xyY) {
    vec3 XYZ;
    XYZ.x = xyY.x * xyY.z / max(xyY.y, 1e-10);
    XYZ.y = xyY.z;
    XYZ.z = (1.0 - xyY.x - xyY.y) * xyY.z / max(xyY.y, 1e-10);
    return XYZ;
}

const float sqrt3over4 = 0.433012701892219;  // sqrt(3.)/4.
const mat3 RGB_2_YAB_MAT = mat3(
    1.0 / 3.0,  1.0 / 2.0,  0.0,
    1.0 / 3.0, -1.0 / 4.0,  sqrt3over4,
    1.0 / 3.0, -1.0 / 4.0, -sqrt3over4
);

vec3 rgb_2_yab(vec3 rgb) {
    vec3 yab = RGB_2_YAB_MAT * rgb;

    return yab;
}

mat3 aces_inverse_mat3x3(mat3 m) {
    float a00 = m[0][0], a01 = m[0][1], a02 = m[0][2];
    float a10 = m[1][0], a11 = m[1][1], a12 = m[1][2];
    float a20 = m[2][0], a21 = m[2][1], a22 = m[2][2];

    float b01 = a22 * a11 - a12 * a21;
    float b11 = -a22 * a10 + a12 * a20;
    float b21 = a21 * a10 - a11 * a20;

    float det = a00 * b01 + a01 * b11 + a02 * b21;

    return mat3(b01, (-a22 * a01 + a02 * a21), (a12 * a01 - a02 * a11),
                b11, (a22 * a00 - a02 * a20), (-a12 * a00 + a02 * a10),
                b21, (-a21 * a00 + a01 * a20), (a11 * a00 - a01 * a10)) / det;
}

vec3 yab_2_rgb(vec3 yab) {
    vec3 rgb = aces_inverse_mat3x3(RGB_2_YAB_MAT) * yab;

    return rgb;
}

vec3 yab_2_ych(vec3 yab) {
    vec3 ych = yab;

    ych[1] = sqrt( pow( yab[1], 2.) + pow( yab[2], 2.) );

    ych[2] = atan( yab[2], yab[1] ) * (180.0 * rPI);
    if (ych[2] < 0.0) ych[2] = ych[2] + 360.;

    return ych;
}

vec3 ych_2_yab(vec3 ych) {
    vec3 yab;
    yab[0] = ych[0];

    float h = ych[2] * (PI / 180.0);
    yab[1] = ych[1]*cos(h);
    yab[2] = ych[1]*sin(h);

    return yab;
}

vec3 rgb_2_ych(vec3 rgb) {
    return yab_2_ych( rgb_2_yab( rgb));
}

vec3 ych_2_rgb(vec3 ych) {
    return yab_2_rgb( ych_2_yab( ych));
}

mat3 calc_sat_adjust_matrix(float sat, vec3 rgb2Y) {
    mat3 M;
    M[0][0] = (1.0 - sat) * rgb2Y[0] + sat;
    M[1][0] = (1.0 - sat) * rgb2Y[0];
    M[2][0] = (1.0 - sat) * rgb2Y[0];
    
    M[0][1] = (1.0 - sat) * rgb2Y[1];
    M[1][1] = (1.0 - sat) * rgb2Y[1] + sat;
    M[2][1] = (1.0 - sat) * rgb2Y[1];
    
    M[0][2] = (1.0 - sat) * rgb2Y[2];
    M[1][2] = (1.0 - sat) * rgb2Y[2];
    M[2][2] = (1.0 - sat) * rgb2Y[2] + sat;

    return M;
}

float center_hue(float hue, float centerH) {
    float hueCentered = hue - centerH;
    if(hueCentered < -180.0)      hueCentered += 360.0;
    else if (hueCentered > 180.0) hueCentered -= 360.0;
    return hueCentered;
}

float uncenter_hue(float hueCentered, float centerH) {
    float hue = hueCentered + centerH;
    if (hue < 0.0)        hue += 360.0;
    else if (hue > 360.0) hue -= 360.0;
    return hue;
}

float cubic_basis_shaper(float x, float w) {
    const mat4 M = mat4(
        -1.0 / 6.0,  3.0 / 6.0, -3.0 / 6.0,  1.0 / 6.0,
         3.0 / 6.0, -6.0 / 6.0,  3.0 / 6.0,  0.0 / 6.0,
        -3.0 / 6.0,  0.0 / 6.0,  3.0 / 6.0,  0.0 / 6.0,
         1.0 / 6.0,  4.0 / 6.0,  1.0 / 6.0,  0.0 / 6.0
    );

    float knots[5] = float[5] (
        w * -0.5,
        w * -0.25,
        0.0,
        w *  0.25,
        w *  0.5
    );

    float y = 0;
    if((x > knots[0]) && (x < knots[4])) {
        float knot_coord = (x - knots[0]) * 4.0 / w;
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec4 monomials = vec4(pow3(t), pow2(t), t, 1.0);

        switch(j) {
            case 3:  y = monomials[0] * M[0][0] + monomials[1] * M[1][0] + monomials[2] * M[2][0] + monomials[3] * M[3][0]; break;
            case 2:  y = monomials[0] * M[0][1] + monomials[1] * M[1][1] + monomials[2] * M[2][1] + monomials[3] * M[3][1]; break;
            case 1:  y = monomials[0] * M[0][2] + monomials[1] * M[1][2] + monomials[2] * M[2][2] + monomials[3] * M[3][2]; break;
            case 0:  y = monomials[0] * M[0][3] + monomials[1] * M[1][3] + monomials[2] * M[2][3] + monomials[3] * M[3][3]; break;
            default: y = 0.0; break;
        }
    }

    return y * 3.0 / 2.0;
}

#endif
