/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _PLANARCLOUDS_
#define _PLANARCLOUDS_

float CalculatePlanarCloudNoise(vec2 position, vec2 wind) {
    position *= rNoiseTexRes;

    const int max_octaves = 6;
    
    const float eScale = 2.0;
    const float eContrib = 0.5;
    const float eOffset = 1.0;

    float scale = 1.0;
    float contrib = 0.5;
    float offset = 0.0;

    float fbm = 0.0;
    float prevNoise = 0.0;

    for (int i = 0; i < max_octaves; ++i) {
        float currentNoise = texture(noisetex, position * scale + (prevNoise * prevNoise * (3.0 - 2.0 * prevNoise)) * contrib * 0.3 + offset - wind).r;
        fbm += currentNoise * contrib;

        prevNoise = currentNoise;

        scale *= eScale;
        contrib *= eContrib;
        offset += eOffset;
        position.y *= eContrib * 1.5 + 1.;
    }

    return fbm;
}

float CalculatePlanarCloudVolume(vec3 position) {
    float worldHeight = position.y - PLANAR_CLOUDS_HEIGHT;
    if (abs(worldHeight) > PLANAR_CLOUDS_THICKNESS) return 0.0;

    vec2 wind = vec2(-TIME) * 0.03;
    vec2 cloudPosition = position.xz * 0.0002667 + wind;

    float localCoverage = texture(noisetex, (TIME * 40.0 + position.xz) * rNoiseTexRes * 0.0001667).x;

    float clouds = CalculatePlanarCloudNoise(cloudPosition, wind * 0.0075);
          clouds = clamp01(clouds * 2.0 - 1.0) * clouds;
          clouds = clouds * clouds * (3.0 - 2.0 * clouds);
          clouds *= pow2(localCoverage);

    return clouds * 0.005;
}

void CalculateMultipleScatteringCloudPhasesPlanar(float vDotL, inout float phases[PLANAR_CLOUDS_MULTISCSAT_QUALITY]){
    float cn = 1.0;

    for (int i = 0; i < PLANAR_CLOUDS_MULTISCSAT_QUALITY; ++i){
        phases[i] = CalculateCloudPhase(vDotL * cn);

        cn *= cloudMultiScattCoeff_c;
    }
}


float CalculatePlanarCloudTransmittanceDepth(vec3 position, const vec3 direction, const int steps) {
    float stepLength = 1.0;

    float totalDepth = 0.0;

    for (int i = 0; i < steps; ++i, position += direction * stepLength) {
        totalDepth += CalculatePlanarCloudVolume(position) * stepLength;
        stepLength *= 2.;
    }

    return totalDepth;
}

void calculatePlanarCloudsScattering(float scatterIntegral, float an, float bn, float transmitDepth, float transmitDepthSky, float phase, inout float directScattering, inout float skylightScattering) {
    float absorb        = exp2(-transmitDepth    * rLOG2 * bn);
    float absorbSky     = exp2(-transmitDepthSky * rLOG2 * bn);
    
    directScattering += scatterIntegral * an * absorb * phase;
    skylightScattering += scatterIntegral * an * absorbSky;
}

void calculatePlanarCloudsScattering(vec3 cloudPosition, vec3 lightDir, float cloudVolume, float transmittance, float phases[PLANAR_CLOUDS_MULTISCSAT_QUALITY], inout float directScattering, inout float skylightScattering) {
    float scatterIntegral = 1.0 - transmittance;

    float transmitDepth = CalculatePlanarCloudTransmittanceDepth(cloudPosition, lightDir, 12);
    float transmitDepthSky = CalculatePlanarCloudTransmittanceDepth(cloudPosition, vec3(0.0, 1.0, 0.0), 8) * 0.2;

    float an = 1.0, bn = 1.0;

        for (int i = 0; i < PLANAR_CLOUDS_MULTISCSAT_QUALITY; ++i) {
            float phase = phases[i];

            calculatePlanarCloudsScattering(scatterIntegral, an, bn, transmitDepth, transmitDepthSky, phase, directScattering, skylightScattering);
        
            an *= cloudMultiScattCoeff_a;
            bn *= cloudMultiScattCoeff_b;
        }
}

vec3 calculatePlanarClouds(const vec3 backgroundColor, const vec3 wDir, const vec3 lightDir, const vec2 planetSphere, const float dither, const float VoL, const float planetRadius) {
        // Early out when the clouds are behind the horizon or not visible.
    if ((eyeAltitude < PLANAR_CLOUDS_HEIGHT && planetSphere.y > 0.0)
        || (eyeAltitude > PLANAR_CLOUDS_HEIGHT && wDir.y > 0.0)) return backgroundColor;

    float[PLANAR_CLOUDS_MULTISCSAT_QUALITY] phases;
    CalculateMultipleScatteringCloudPhasesPlanar(VoL, phases);

    vec2 cloudSphere = rsi(vec3(0.0, planetRadius + eyeAltitude, 0.0), wDir, planetRadius + PLANAR_CLOUDS_HEIGHT);
    float cloudDistance = eyeAltitude > PLANAR_CLOUDS_HEIGHT ? cloudSphere.x : cloudSphere.y;

    if (cloudDistance <= 0.0) return backgroundColor;

    vec3 cloudPosition = cloudDistance * wDir + cameraPosition;
         cloudPosition.y = PLANAR_CLOUDS_HEIGHT;
    
    float cloudVolume = CalculatePlanarCloudVolume(cloudPosition);
    float transmittance = exp2(-cloudVolume * rLOG2 * PLANAR_CLOUDS_THICKNESS);

    float directScattering = 0.0;
    float skylightScattering = 0.0;

    calculatePlanarCloudsScattering(cloudPosition, lightDir, cloudVolume, transmittance, phases, directScattering, skylightScattering);

    vec3 directLighting = directScattering * sunIlluminanceClouds;
    vec3 skyLighting = skylightScattering * skyIlluminanceClouds * 0.25 * rPI;
    vec3 scattering = directLighting + skyLighting;

    vec3 kCamera = vec3(0.0, 0.05 + cameraPosition.y * 0.001 + ATMOSPHERE.bottom_radius, 0.0);
	vec3 kPoint  = (cloudPosition - cameraPosition) * 0.001 + kCamera;
	
    vec3 transmitAP;
    vec3 in_scatter_moon;
    vec3 in_scatter = GetSkyRadianceToPoint(ATMOSPHERE, colortex7, colortex7, colortex7, kCamera, kPoint, 0.0, wSunVector, transmitAP, in_scatter_moon) * SKY_SPECTRAL_RADIANCE_TO_LUMINANCE;
    in_scatter_moon *= SKY_SPECTRAL_RADIANCE_TO_LUMINANCE_MOON;
    in_scatter += in_scatter_moon;

	if (any(isnan(in_scatter))) {
		in_scatter = vec3(0.0);
		transmitAP = vec3(1.0);
	}

    scattering = scattering * transmitAP + in_scatter * (1.0 - transmittance);

    return backgroundColor * transmittance + scattering;
}
#endif