/********************************************************
    © 2020 Continuum Graphics LLC. All Rights Reserved
 ********************************************************/

#if !defined _WATERWAVES_
#define _WATERWAVES_

#define DRAG_MULT 0.25
#define WAVE_HEIGHT 0.25
#define OCTAVES 16

vec2 wavedx(vec2 position, vec2 direction, float speed, float frequency, float timeshift) {
    float x    = dot(direction, position) * frequency + timeshift * speed;
    float wave = exp(sin(x) - 1.0);
    float dx   = wave * cos(x);
    
    return vec2(wave, -dx);
}

vec2 calculateInterpolated2DNoise(vec2 p) {
    vec2 pos = p * noiseTextureResolution;

    vec2 fr = fract(pos);
    fr = fr * fr * (3.0 - 2.0 * fr);
    vec2 fl = (floor(pos) - 0.5) * rNoiseTexRes;

    const vec2 delta = vec2(rNoiseTexRes, 0.0);

    vec2 ix = mix(texture2D(noisetex, fl).xy,            texture2D(noisetex, fl + delta.xy).xy, fr.x);
    vec2 iy = mix(texture2D(noisetex, fl + delta.yx).xy, texture2D(noisetex, fl + delta.xx).xy, fr.x);

    return mix(ix, iy, fr.y);
}

float getwaves(vec2 position, float roughnessDistanceEstimate) {
    vec3 noise;
    noise.x  = calculateInterpolated2DNoise((position - TIME) * 0.001).x ;
    noise.yz = calculateInterpolated2DNoise((position - TIME) * 0.005).xy;
    float heightMult = (clamp01(noise.x * 2.0 - 0.2) * 0.5 + 0.5);
    
    float iter   = 0.0;
    float phase  = 1.0;
    float speed  = 2.0;
    float weight = 1.0;
    float w      = 0.0;
    float ws     = 0.0;
    for (int i = 0; i < OCTAVES; i++) {
        vec2 p = vec2( sin(iter),  cos(iter) );
        vec2 res = wavedx(p * noise.yz + position, p, speed, phase, TIME);
        position += p * res.y * weight * DRAG_MULT * heightMult;
        w += res.x * weight;
        iter += 12.0;
        ws += weight;
        weight = mix(weight, 0.0, 0.2);
        phase *= 1.18;
        speed *= 1.07;
    }

    float heightMix = 1.0 - roughnessDistanceEstimate * 0.86;
    
    return (w / ws * heightMult - 1.0) * WAVE_HEIGHT * heightMix;
}

vec3 calculateWaterNormal(vec3 position, float roughnessDistanceEstimate){
    const float d = 0.1;
    const vec3  delta = vec3(d,  1.0 / d,  0.0);

    float d0 = getwaves(position.xz, roughnessDistanceEstimate);
    float dx = getwaves(position.xz + delta.xz, roughnessDistanceEstimate);
    float dy = getwaves(position.xz + delta.zx, roughnessDistanceEstimate);

    dx = (d0 - dx) * delta.y;
    dy = (d0 - dy) * delta.y;

    return normalize(vec3(dx, dy, 1.0));
}

/*
vec3 calculateWaterNormal(vec3 position, float roughnessDistanceEstimate){
    const float d = 0.1;
    const vec3  delta = vec3(d,  1.0 / d,  0.0);

    float d0 = getwaves(position.xz, roughnessDistanceEstimate);
    float dx = getwaves(position.xz - delta.xz, roughnessDistanceEstimate);
    float dy = getwaves(position.xz - delta.zx, roughnessDistanceEstimate);
    float dx0 = getwaves(position.xz + delta.xz, roughnessDistanceEstimate);
    float dy0 = getwaves(position.xz + delta.zx, roughnessDistanceEstimate);

    dx = ((dx - d0) + (d0 - dx0)) * delta.y;
    dy = ((dy - d0) + (d0 - dy0)) * delta.y;

    vec3 normal = vec3(dx, dy, 2.0 - dx * dx - dy * dy);

    return normalize(normal);
}
*/

#endif
