/*
 * Decompiled with CFR 0.152.
 */
package journeymap.client.cartography;

import com.google.common.collect.ComparisonChain;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Ordering;
import com.google.common.io.Files;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.annotations.Since;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import journeymap.client.Constants;
import journeymap.client.JourneymapClient;
import journeymap.client.cartography.RGB;
import journeymap.client.io.FileHandler;
import journeymap.client.log.ChatLog;
import journeymap.client.model.BlockMD;
import journeymap.common.Journeymap;
import journeymap.common.log.LogFormatter;
import net.minecraft.block.Block;
import net.minecraft.client.Minecraft;
import net.minecraftforge.fml.client.FMLClientHandler;
import org.apache.logging.log4j.core.helpers.Strings;

public class ColorPalette {
    public static final String HELP_PAGE = "http://journeymap.info/Color_Palette";
    public static final String SAMPLE_STANDARD_PATH = ".minecraft/journeymap/";
    public static final String SAMPLE_WORLD_PATH = ".minecraft/journeymap/data/*/worldname/";
    public static final String JSON_FILENAME = "colorpalette.json";
    public static final String HTML_FILENAME = "colorpalette.html";
    public static final String VARIABLE = "var colorpalette=";
    public static final Charset UTF8 = Charset.forName("UTF-8");
    public static final double VERSION = 5.2;
    public static final Gson GSON = new GsonBuilder().setVersion(5.2).setPrettyPrinting().create();
    @Since(value=3.0)
    double version;
    @Since(value=1.0)
    String name;
    @Since(value=1.0)
    String generated;
    @Since(value=1.0)
    String[] description;
    @Since(value=1.0)
    boolean permanent;
    @Since(value=1.0)
    String resourcePacks;
    @Since(value=2.0)
    String modNames;
    @Since(value=5.2)
    ArrayList<BlockColor> blockColors = new ArrayList(0);
    private transient File origin;

    ColorPalette() {
    }

    private ColorPalette(String resourcePacks, String modNames) {
        this.version = 5.2;
        this.name = Constants.getString("jm.colorpalette.file_title");
        this.generated = String.format("Generated using %s for %s on %s", JourneymapClient.MOD_NAME, "1.8.9", new Date());
        this.resourcePacks = resourcePacks;
        this.modNames = modNames;
        ArrayList<String> lines = new ArrayList<String>();
        lines.add(Constants.getString("jm.colorpalette.file_header_1"));
        lines.add(Constants.getString("jm.colorpalette.file_header_2", HTML_FILENAME));
        lines.add(Constants.getString("jm.colorpalette.file_header_3", JSON_FILENAME, SAMPLE_WORLD_PATH));
        lines.add(Constants.getString("jm.colorpalette.file_header_4", JSON_FILENAME, SAMPLE_STANDARD_PATH));
        lines.add(Constants.getString("jm.config.file_header_5", HELP_PAGE));
        this.description = lines.toArray(new String[4]);
        this.blockColors = this.deriveBlockColors();
    }

    public static ColorPalette getActiveColorPalette() {
        File standardPaletteFile;
        ColorPalette palette;
        String resourcePacks = Constants.getResourcePackNames();
        String modNames = Constants.getModNames();
        File worldPaletteFile = ColorPalette.getWorldPaletteFile();
        if (worldPaletteFile.canRead() && (palette = ColorPalette.loadFromFile(worldPaletteFile)) != null) {
            if (palette.version < 5.2) {
                Journeymap.getLogger().warn(String.format("Existing world color palette is obsolete. Required version: %s.  Found version: %s", 5.2, palette.version));
            } else {
                return palette;
            }
        }
        if ((standardPaletteFile = ColorPalette.getStandardPaletteFile()).canRead()) {
            ColorPalette palette2 = ColorPalette.loadFromFile(standardPaletteFile);
            if (palette2 != null && palette2.version != 5.2) {
                Journeymap.getLogger().warn(String.format("Existing color palette is unusable. Required version: %s.  Found version: %s", 5.2, palette2.version));
                standardPaletteFile.renameTo(new File(standardPaletteFile.getParentFile(), standardPaletteFile.getName() + ".v" + palette2.version));
                palette2 = null;
            }
            if (palette2 != null) {
                if (palette2.isPermanent()) {
                    Journeymap.getLogger().info("Existing color palette is set to be permanent.");
                    return palette2;
                }
                if (resourcePacks.equals(palette2.resourcePacks)) {
                    if (modNames.equals(palette2.modNames)) {
                        Journeymap.getLogger().info("Existing color palette's resource packs and mod names match current loadout.");
                        return palette2;
                    }
                    Journeymap.getLogger().warn("Existing color palette's mods no longer match current loadout.");
                    Journeymap.getLogger().info(String.format("WAS: %s\nNOW: %s", palette2.modNames, modNames));
                } else {
                    Journeymap.getLogger().warn("Existing color palette's resource packs no longer match current loadout.");
                    Journeymap.getLogger().info(String.format("WAS: %s\nNOW: %s", palette2.resourcePacks, resourcePacks));
                }
            }
        }
        return null;
    }

    public static ColorPalette create(boolean standard, boolean permanent) {
        long start = System.currentTimeMillis();
        ColorPalette palette = null;
        try {
            String resourcePackNames = Constants.getResourcePackNames();
            String modPackNames = Constants.getModNames();
            palette = new ColorPalette(resourcePackNames, modPackNames);
            palette.setPermanent(permanent);
            palette.writeToFile(standard);
            long elapsed = System.currentTimeMillis() - start;
            Journeymap.getLogger().info(String.format("Color palette file generated with %d colors in %dms for: %s", palette.size(), elapsed, palette.getOrigin()));
            return palette;
        }
        catch (Exception e) {
            Journeymap.getLogger().error("Couldn't create ColorPalette: " + LogFormatter.toString(e));
            return null;
        }
    }

    private static File getWorldPaletteFile() {
        Minecraft mc = FMLClientHandler.instance().getClient();
        return new File(FileHandler.getJMWorldDir(mc), JSON_FILENAME);
    }

    private static File getStandardPaletteFile() {
        return new File(FileHandler.getJourneyMapDir(), JSON_FILENAME);
    }

    private static ColorPalette loadFromFile(File file) {
        String jsonString = null;
        try {
            jsonString = Files.toString((File)file, (Charset)UTF8).replaceFirst(VARIABLE, "");
            ColorPalette palette = (ColorPalette)GSON.fromJson(jsonString, ColorPalette.class);
            palette.origin = file;
            palette.getOriginHtml(true, true);
            return palette;
        }
        catch (Throwable e) {
            ChatLog.announceError(Constants.getString("jm.colorpalette.file_error", file.getPath()));
            try {
                file.renameTo(new File(file.getParentFile(), file.getName() + ".bad"));
            }
            catch (Exception e2) {
                Journeymap.getLogger().error("Couldn't rename bad palette file: " + e2);
            }
            return null;
        }
    }

    private String substituteValueInContents(String contents, String key, Object ... params) {
        String token = String.format("\\$%s\\$", key);
        return contents.replaceAll(token, Matcher.quoteReplacement(Constants.getString(key, params)));
    }

    private ArrayList<BlockColor> deriveBlockColors() {
        HashBasedTable blockColorTable = HashBasedTable.create((int)512, (int)16);
        for (BlockMD blockMD : BlockMD.getAll()) {
            if (blockMD.isAir()) continue;
            if (blockMD.hasFlag(BlockMD.Flag.Error)) {
                Journeymap.getLogger().debug("Block with Error flag won't be saved to color palette: " + blockMD);
                continue;
            }
            if (Strings.isEmpty((CharSequence)blockMD.getUid())) {
                Journeymap.getLogger().debug("Block without uid won't be saved to color palette: " + blockMD);
                continue;
            }
            Integer color = blockMD.getColor();
            if (color == null) {
                Journeymap.getLogger().debug("Block without color won't be saved to color palette: " + blockMD);
                continue;
            }
            Block block = blockMD.getBlock();
            if (block == null) {
                Journeymap.getLogger().debug("Bad block won't be saved to color palette: " + blockMD);
                continue;
            }
            blockColorTable.put((Object)blockMD.getBlock(), (Object)color, (Object)blockMD);
        }
        ArrayList<BlockColor> list = new ArrayList<BlockColor>(512);
        for (Block block : blockColorTable.rowKeySet()) {
            BlockColor blockColors = null;
            for (Map.Entry entry : blockColorTable.row((Object)block).entrySet()) {
                if (blockColors == null) {
                    blockColors = new BlockColor((BlockMD)entry.getValue(), (Integer)entry.getKey());
                    list.add(blockColors);
                    continue;
                }
                blockColors.addVariant((BlockMD)entry.getValue(), (Integer)entry.getKey());
            }
            if (blockColors == null) continue;
            blockColors.sort();
        }
        Collections.sort(list);
        return list;
    }

    private boolean writeToFile(boolean standard) {
        File palleteFile = null;
        try {
            palleteFile = standard ? ColorPalette.getStandardPaletteFile() : ColorPalette.getWorldPaletteFile();
            Files.write((CharSequence)(VARIABLE + GSON.toJson((Object)this)), (File)palleteFile, (Charset)UTF8);
            this.origin = palleteFile;
            this.getOriginHtml(true, true);
            return true;
        }
        catch (Exception e) {
            Journeymap.getLogger().error(String.format("Can't save color palette file %s: %s", palleteFile, LogFormatter.toString(e)));
            return false;
        }
    }

    void updateColors() {
        for (BlockColor blockColor : this.blockColors) {
            blockColor.inflate().updateBlockMD();
        }
    }

    public File getOrigin() throws IOException {
        return this.origin.getCanonicalFile();
    }

    public File getOriginHtml(boolean createIfMissing, boolean overwriteExisting) {
        try {
            if (this.origin == null) {
                return null;
            }
            File htmlFile = new File(this.origin.getParentFile(), HTML_FILENAME);
            if (!htmlFile.exists() && createIfMissing || overwriteExisting) {
                htmlFile = FileHandler.copyColorPaletteHtmlFile(this.origin.getParentFile(), HTML_FILENAME);
                String htmlString = Files.toString((File)htmlFile, (Charset)UTF8);
                htmlString = this.substituteValueInContents(htmlString, "jm.colorpalette.file_title", new Object[0]);
                htmlString = this.substituteValueInContents(htmlString, "jm.colorpalette.file_missing_data", JSON_FILENAME);
                htmlString = this.substituteValueInContents(htmlString, "jm.colorpalette.resource_packs", new Object[0]);
                htmlString = this.substituteValueInContents(htmlString, "jm.colorpalette.mods", new Object[0]);
                htmlString = this.substituteValueInContents(htmlString, "jm.colorpalette.basic_colors", new Object[0]);
                htmlString = this.substituteValueInContents(htmlString, "jm.colorpalette.biome_colors", new Object[0]);
                Files.write((CharSequence)htmlString, (File)htmlFile, (Charset)UTF8);
            }
            return htmlFile;
        }
        catch (Throwable t) {
            Journeymap.getLogger().error("Can't get colorpalette.html: " + t);
            return null;
        }
    }

    public boolean isPermanent() {
        return this.permanent;
    }

    public void setPermanent(boolean permanent) {
        this.permanent = permanent;
    }

    public boolean isStandard() {
        return this.origin != null && this.origin.getParentFile().getAbsoluteFile().equals(FileHandler.getJourneyMapDir().getAbsoluteFile());
    }

    public int size() {
        int size = 0;
        for (BlockColor blockColor : this.blockColors) {
            size += blockColor.size();
        }
        return size;
    }

    public String toString() {
        return "ColorPalette[" + this.resourcePacks + "]";
    }

    class BlockColor
    implements Comparable<BlockColor> {
        @Since(value=5.2)
        String uid;
        @Since(value=5.2)
        String name;
        @Since(value=5.2)
        String color;
        @Since(value=5.2)
        Float alpha;
        @Since(value=5.2)
        Integer meta;
        @Since(value=5.2)
        ArrayList<BlockColor> variants;

        BlockColor() {
        }

        BlockColor(BlockMD blockMD, Integer color) {
            this.uid = blockMD.getUid();
            this.name = blockMD.getName();
            this.color = RGB.toHexString(color);
            this.alpha = Float.valueOf(blockMD.getAlpha());
            this.meta = blockMD.getMeta();
        }

        void addVariant(BlockMD blockMD, Integer color) {
            if (this.variants == null) {
                this.variants = new ArrayList();
            }
            BlockColor variant = new BlockColor(blockMD, color);
            this.deflate(variant);
            this.variants.add(variant);
        }

        void deflate() {
            if (this.variants != null) {
                for (BlockColor variant : this.variants) {
                    this.deflate(variant);
                }
            }
        }

        void deflate(BlockColor variant) {
            if (Objects.equals(this.uid, variant.uid)) {
                variant.uid = null;
            }
            if (Objects.equals(this.name, variant.name)) {
                variant.name = null;
            }
            if (Objects.equals(this.color, variant.color)) {
                variant.color = null;
            }
            if (Objects.equals(this.alpha, variant.alpha)) {
                variant.alpha = null;
            }
            variant.deflate();
        }

        BlockColor inflate() {
            if (this.variants != null) {
                for (BlockColor variant : this.variants) {
                    this.inflate(variant);
                }
            }
            return this;
        }

        void inflate(BlockColor variant) {
            if (variant.uid == null) {
                variant.uid = this.uid;
            }
            if (variant.name == null) {
                variant.name = this.name;
            }
            if (variant.color == null) {
                variant.color = this.color;
            }
            if (variant.alpha == null) {
                variant.alpha = this.alpha;
            }
            variant.inflate();
        }

        void updateBlockMD() {
            if (!this.isInflated()) {
                throw new IllegalArgumentException("BlockColor object must be inflated before use.");
            }
            BlockMD blockMD = BlockMD.get(this.uid, this.meta);
            if (blockMD == null) {
                Journeymap.getLogger().warn(String.format("Block referenced in Color Palette is not registered: %s:%s ", this.uid, this.meta));
                return;
            }
            if (blockMD.hasFlag(BlockMD.Flag.Transparency)) {
                blockMD.setAlpha(this.alpha != null ? this.alpha.floatValue() : 1.0f);
            }
            int color = RGB.hexToInt(this.color);
            blockMD.setColor(color);
            if (this.variants != null) {
                for (BlockColor variant : this.variants) {
                    variant.updateBlockMD();
                }
            }
        }

        boolean isInflated() {
            return this.uid != null && this.meta != null && this.alpha != null && this.color != null;
        }

        @Override
        public int compareTo(BlockColor that) {
            Ordering ordering = Ordering.natural().nullsLast();
            return ComparisonChain.start().compare((Object)this.uid, (Object)that.uid, (Comparator)ordering).compare((Object)this.meta, (Object)that.meta, (Comparator)ordering).compare((Object)this.name, (Object)that.name, (Comparator)ordering).compare((Object)this.color, (Object)that.color, (Comparator)ordering).compare((Object)this.alpha, (Object)that.alpha, (Comparator)ordering).result();
        }

        public int size() {
            int size = 1;
            if (this.variants != null) {
                for (BlockColor variant : this.variants) {
                    size += variant.size();
                }
            }
            return size;
        }

        public void sort() {
            if (this.variants != null) {
                Collections.sort(this.variants);
            }
        }
    }
}

