#version 400 compatibility

/*
Copyright (C) 2019 RRe36

All Rights Reserved unless otherwise explicitly stated.


By downloading this you have agreed to the license and terms of use.
These can be found inside the included license-file or here: https://github.com/rre36/glsl_kappa/blob/master/LICENSE

Violating these terms may be penalized with actions according to the Digital Millennium Copyright Act (DMCA), the Information Society Directive and/or similar laws depending on your country.
*/

#include "/lib/common.glsl"

in vec2 coord;

flat in float light_flip;

flat in mat4x3 light_color;

uniform sampler2D colortex0;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D colortex5;

uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

uniform sampler2D depthtex2;

uniform sampler2D noisetex;

uniform sampler2DShadow shadowtex0;
uniform sampler2DShadow shadowtex1;
uniform sampler2D shadowcolor0;

uniform int isEyeInWater;
uniform int frameCounter;

uniform float eyeAltitude;
uniform float frameTimeCounter;
uniform float sunAngle;
uniform float wetness;

uniform ivec2 eyeBrightness;
uniform ivec2 eyeBrightnessSmooth;

uniform vec2 pixelSize;
uniform vec2 taaOffset;

uniform vec3 lightvec, lightvecView;
uniform vec3 cameraPosition;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;
uniform mat4 shadowModelView, shadowModelViewInverse;
uniform mat4 shadowProjection, shadowProjectionInverse;

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    screenpos.xy -= taaOffset;

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

int decodeMatID16(float x) {
    return int(x*65535.0);
}

#include "/lib/frag/bluenoise.glsl"

/* ------ volumetric fog ------ */

#include "/lib/light/warp.glsl"

vec3 get_shadowcoord(vec3 viewpos) {  //shadow 2d
    vec3 pos    = viewpos;
        pos    += vec3(0.08)*lightvec;
        pos     = viewMAD(shadowModelView, pos);
        pos     = projMAD(shadowProjection, pos);
        pos.z  *= 0.2;

        pos.xy  = warp_shadowmap(pos.xy);

    return pos*0.5+0.5;
}

vec3 get_shadowcol(sampler2D tex, vec2 coord) {
    vec4 x  = texture(tex, coord);
    return mix(vec3(1.0), x.rgb, x.a);
}

#include "/lib/frag/noise.glsl"

const float fog_alt = 90.0;
const float fog_maxalt = fog_alt+60.0;
const float fog_clip = 384.0;
float fog_density = mix(1.0*daytime.x + 0.33*daytime.y + 0.24*daytime.z + 0.5*daytime.w, 2.9, wetness);
float fog_tick  = frameTimeCounter*0.2;

vec3 get_fog_density(vec3 pos) {
    float fade  = 1.0-sstep(pos.y, fog_alt, fog_maxalt);
    vec3 wind   = vec3(fog_tick, fog_tick*0.1, fog_tick*0.03);

    pos    += wind;
    pos    *= 0.03;
    pos.y *= 1.7;

    float shape = 0.0;

    #ifdef vfog_clouds
        if (daytime.y < 0.99) {
            shape   = value_3d(pos);
            shape  += value_3d(pos * 3.0) * 0.5;
            shape  *= rcp(1.5);
            shape  -= 0.2;
            shape  *= 1.9;
            shape   = mix(shape, 1.0, sqr(daytime.y));
        } else {
            shape   = 1.0;
        }
    #else
        shape = 1.0;
    #endif

        shape  *= fade;
        shape  -= 0.02;

    vec3 ret = vec3(max(shape*fog_density*0.04, 0.0));
        ret.x *= 1.0-wetness*0.5;
        ret.x *= 1.0 + sqr(daytime.x) * 0.4;
        ret.y *= 0.85;
        ret.z  = 0.0;

    return ret*vfog_density;
}

#include "/lib/atmos/phase.glsl"

#ifdef vfog_bilateral
    #define vf_bstep int(12*vfog_quality)
    #define vf_dstep 12
#else
    #define vf_bstep int(14*vfog_quality)
    #define vf_dstep 14
#endif

const float mie_g   = 0.7;

const vec3 rayleigh_coeff = vec3(2e-6, 5.7e-6, 8e-6);
const vec3 mie_coeff    = vec3(7.8e-6);    //19e-5 for rain

const mat2x3 scattering_mat   = mat2x3(rayleigh_coeff, mie_coeff);
const mat2x3 extinct_mat      = mat2x3(rayleigh_coeff, mie_coeff);

vec2 phaseFunction(float cosTheta) { 
    return vec2(rayleigh_phase(cosTheta), hg_mie(cosTheta, mie_g*0.93) + 0.06);
}

void compute_fog(out mat2x3 data, vec3 scenepos0, vec3 scenepos1, vec3 svec, bool sky, float vdotl) {
    //const int steps = 28;
    const float air_density = 5.7e4;

    float ts        = length(svec*((fog_maxalt-eyeAltitude)/svec.y));
    float bs        = length(svec*((-32.0-eyeAltitude)/svec.y));

    float sd        = svec.y>0.0 ? ts : bs;

    float edist     = sky ? min(sd, fog_clip) : length(scenepos0);
    float sdist     = eyeAltitude>fog_maxalt ? ts : 1.0;

    vec3 spos       = eyeAltitude>fog_maxalt ? sdist*svec + gbufferModelViewInverse[3].xyz : gbufferModelViewInverse[3].xyz;
    vec3 epos       = sky ? edist*svec : scenepos0;

    if (isEyeInWater == 1) {
        spos        = scenepos0;
        epos        = sky ? epos : scenepos1;
    }

    float dither    = dither_bluenoise();

    float bstep      = length((epos-spos));
    float stepcoeff  = saturate(bstep/256.0);

    int steps        = vf_bstep+int(stepcoeff*vf_dstep);

    vec3 rstep      = (epos-spos)/steps;
    vec3 rpos       = spos + rstep*dither;
    float rlength   = length(rstep);

    mat2x3 scatter  = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    vec2 phase      = phaseFunction(vdotl);
    float phaseIso  = 0.5*rcp(pi);

    float cave_fix  = linStep(eyeBrightness.y/240.0, 0.1, 0.9);

    vec3 sunlight   = (sunAngle<0.5 ? light_color[0] : light_color[3])*light_flip * pi;
    vec3 skylight   = light_color[1]*cave_fix;

    for (int i = 0; i<steps; ++i, rpos += rstep) {
        if (max3(transmittance) < 0.01) break;
        if ((rpos.y+cameraPosition.y)>fog_maxalt) continue;
        vec3 density   = get_fog_density(rpos+cameraPosition);
        if (max3(density) <= 0.0) continue;

        vec3 airmass = density * rlength * air_density;
        vec3 od     = extinct_mat * (airmass.xy);

        vec3 stept  = saturate(expf(-od * 0.125));
        vec3 integ  = saturate((stept - 1.0) / -max(od, 1e-16));
        vec3 vs     = transmittance * integ;

        vec3 sun_s  = scattering_mat * (airmass.xy * phase) * vs;
        vec3 sky_s  = scattering_mat * (airmass.xy * phaseIso) * vs;

        vec3 shadowpos = get_shadowcoord(rpos);

        float s0        = shadow2D(shadowtex0, shadowpos).x;
        float shadow    = 1.0;
        vec3 shadowcol  = vec3(1.0);

        if (s0<1.0) {
            shadow      = shadow2D(shadowtex1, shadowpos).x;

            if (distance(shadow, s0)>0.1) {
                shadowcol = get_shadowcol(shadowcolor0, shadowpos.xy);
            } else {
                shadowcol = vec3(1.0);
            }
        }

        scatter[0]      += sun_s * shadow * transmittance * shadowcol;
        scatter[1]      += sky_s * transmittance;
        transmittance  *= stept;
    }

    vec3 color      = scatter[0]*sunlight + scatter[1]*skylight;

    data[0]         = color;
    data[1]         = transmittance;
}

void compute_fog2(inout mat2x3 data, vec3 scenepos0, vec3 scenepos1, vec3 svec, bool sky, float vdotl) {
    //const int steps = 28;
    const float air_density = 5.7e4;

    float ts        = length(svec*((fog_maxalt-eyeAltitude)/svec.y));
    float bs        = length(svec*((-32.0-eyeAltitude)/svec.y));

    float sd        = svec.y>0.0 ? ts : bs;

    float edist     = sky ? min(sd, fog_clip) : length(scenepos0);
    float sdist     = eyeAltitude>fog_maxalt ? ts : length(scenepos1);

    vec3 spos       = eyeAltitude>fog_maxalt ? sdist*svec : scenepos0;
    vec3 epos       = sky ? edist*svec : scenepos1;

    float dither    = dither_bluenoise();

    float bstep      = length((epos-spos));
    float stepcoeff  = saturate(bstep/256.0);

    int steps        = int(vf_bstep * 0.66)+int(stepcoeff * vf_dstep * 0.5);

    vec3 rstep      = (epos-spos)/steps;
    vec3 rpos       = spos + rstep*dither;
    float rlength   = length(rstep);

    mat2x3 scatter  = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    vec2 phase      = phaseFunction(vdotl);
    float phaseIso  = 0.5*rcp(pi);

    float cave_fix  = linStep(eyeBrightness.y/240.0, 0.1, 0.9);

    vec3 sunlight   = (sunAngle<0.5 ? light_color[0] : light_color[3])*light_flip * pi;
    vec3 skylight   = light_color[1]*cave_fix;

    for (int i = 0; i<steps; ++i, rpos += rstep) {
        if (max3(transmittance) < 0.01) break;
        if ((rpos.y+cameraPosition.y)>fog_maxalt) continue;
        vec3 density   = get_fog_density(rpos+cameraPosition);
        if (max3(density) <= 0.0) continue;

        vec3 airmass = density * rlength * air_density;
        vec3 od     = extinct_mat * (airmass.xy);

        vec3 stept  = saturate(expf(-od * 0.125));
        vec3 integ  = saturate((stept - 1.0) / -max(od, 1e-16));
        vec3 vs     = transmittance * integ;

        vec3 sun_s  = scattering_mat * (airmass.xy * phase) * vs;
        vec3 sky_s  = scattering_mat * (airmass.xy * phaseIso) * vs;

        vec3 shadowpos = get_shadowcoord(rpos);

        float s0        = shadow2D(shadowtex0, shadowpos).x;
        float shadow    = 1.0;
        vec3 shadowcol  = vec3(1.0);

        if (s0<1.0) {
            shadow      = shadow2D(shadowtex1, shadowpos).x;

            if (distance(shadow, s0)>0.1) {
                shadowcol = get_shadowcol(shadowcolor0, shadowpos.xy);
            } else {
                shadowcol = vec3(1.0);
            }
        }

        scatter[0]      += sun_s * shadow * transmittance * shadowcol;
        scatter[1]      += sky_s * transmittance;
        transmittance  *= stept;
    }

    vec3 color      = scatter[0]*sunlight + scatter[1]*skylight;

    data[0]        += color;
    data[1]        *= transmittance;
}


void compute_water(out mat2x3 data, vec3 scenepos0, vec3 scenepos1, vec3 svec, float vdotl) {
    const float water_density = 5e-1 * vwater_density;

    bool in_water   = isEyeInWater == 1;

    vec3 epos       = in_water ? scenepos0 : scenepos1;
    vec3 spos       = in_water ? gbufferModelViewInverse[3].xyz : scenepos0;

    float bstep      = length((epos-spos));
    float stepcoeff  = saturate(bstep/64.0);
    int steps        = 6;

    if (isEyeInWater == 1) steps = 8+int(stepcoeff*8*vwater_quality);
    else steps = 4+int(stepcoeff*6*vwater_quality);

    float dither    = dither_bluenoise();

    vec3 rstep      = (epos-spos)/steps;
    vec3 rpos       = spos + rstep*dither;
    float rlength   = length(rstep);

    mat2x3 scatter  = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    float phase     = hg_mie(vdotl, 0.76);
    float phaseIso  = 0.25*rcp(pi);
        phase       = phaseIso*0.2 + phase*0.8;

    float cave_fix  = linStep(eyeBrightnessSmooth.y/240.0, 0.1, 0.9);
        cave_fix    = max(cave_fix, float(in_water)*0.1);

    vec3 sunlight   = (sunAngle<0.5 ? light_color[0]*pi : light_color[3])*light_flip;
    vec3 skylight   = light_color[1]*(cave_fix);

    #ifdef custom_water_color
        const vec3 extinct_coeff = vec3(9e-1, 1e-1, 7e-2);
        const vec3 scatter_coeff = vec3(2e-1, 3e-1, 5e-1) * 0.7;
    #else
        const vec3 extinct_coeff = vec3(9e-1, 3e-1, 3e-2);
        const vec3 scatter_coeff = vec3(10e-2, 2e-1, 5e-1) * 0.66;
    #endif

    for (int i = 0; i<steps; ++i, rpos += rstep) {
        if (max3(transmittance) < 0.01) break;

        float watermass = water_density * rlength;
        vec3 od     = extinct_coeff * watermass;

        vec3 stept  = saturate(expf(-od));
        vec3 integ  = saturate((stept - 1.0) / -od);
        vec3 vs     = transmittance * integ;

        vec3 sun_s  = scatter_coeff * watermass * phase * vs;
        vec3 sky_s  = scatter_coeff * watermass * phaseIso * vs;

        vec3 shadowpos = get_shadowcoord(rpos);

        float shadow    = shadow2D(shadowtex1, shadowpos).x;

        scatter[0]      += sun_s * shadow * transmittance;
        scatter[1]      += sky_s * transmittance;
        transmittance  *= stept;
    }
    transmittance   = linStep(transmittance, 0.01, 1.0);

    vec3 color      = scatter[0]*sunlight + scatter[1]*skylight;

    data[0]         = color;
    data[1]         = transmittance;
}

float min_depth3x3(sampler2D depthtex, vec2 coord, vec2 px) {

    float tl    = texture(depthtex, coord + vec2(-px.x, -px.y)).x;
    float tc    = texture(depthtex, coord + vec2(0.0, -px.y)).x;
    float tr    = texture(depthtex, coord + vec2(px.x, -px.y)).x;
    float tmin  = min(tl, min(tc, tr));

    float ml    = texture(depthtex, coord + vec2(-px.x, 0.0)).x;
    float mc    = texture(depthtex, coord).x;
    float mr    = texture(depthtex, coord + vec2(px.x, 0.0)).x;
    float mmin  = min(ml, min(mc, mr));

    float bl    = texture(depthtex, coord + vec2(-px.x, px.y)).x;
    float bc    = texture(depthtex, coord + vec2(0.0, px.y)).x;
    float br    = texture(depthtex, coord + vec2(px.x, px.y)).x;
    float bmin  = min(bl, min(bc, br));

    return min(tmin, min(mmin, bmin));
}

void main() {
    vec4 scenecol   = stex(colortex0);  //that macro certainly makes it neater
    //vec4 tex1       = stex(colortex1);
    //vec4 tex2       = stex(colortex2);
    //vec4 tex3       = stex(colortex3);
    vec4 tex5       = stex(colortex5);
    vec4 tex6       = vec4(0.0, 0.0, 0.0, 1.0);

    tex6.rgb    = vec3(1.0);

    #ifdef vfog_enabled
    //atmosfog
    vec2 fogcoord   = (coord - vec2(0.5, 0.0))*2.0;
    if (fogcoord == clamp(fogcoord, -0.003, 1.003)) {
        vec2 coord      = saturate(fogcoord);

        float scenedepth0 = stex(depthtex0).r;
        float scenedepth1 = stex(depthtex1).r;

        vec3 viewpos0   = screen_viewspace(vec3(coord, scenedepth0));
        vec3 viewpos1   = screen_viewspace(vec3(coord, scenedepth1));

        vec3 scenepos0  = view_scenespace(viewpos0);
        vec3 scenepos1  = view_scenespace(viewpos1);

        vec3 sdir       = normalize(scenepos0);

        float vdotl     = dot(normalize(viewpos0), lightvecView);

        mat2x3 atmos;
            atmos[0] = vec3(0.0);
            atmos[1] = vec3(1.0);

        compute_fog(atmos, scenepos0, scenepos1, sdir, !landMask(scenedepth0), vdotl);
        tex5.rgb    = atmos[0];
        tex6.rgb    = atmos[1];
    }
    #endif

    #if (defined vwater_enabled || defined vfog_translucents)
    //water volume
    vec2 fogcoordw  = (coord - vec2(0.0, 0.5))*2.02;
    if (fogcoordw == clamp(fogcoordw, -0.003, 1.003)) {
        vec2 coord      = saturate(fogcoordw);
        vec4 tex2       = stex(colortex2);
        int mat_id      = decodeMatID16(tex2.x);

        float scenedepth0 = min_depth3x3(depthtex0, coord, pixelSize);
        float scenedepth1 = stex(depthtex1).r;

        bool translucent = (scenedepth0<scenedepth1);

        if (translucent || isEyeInWater==1) {
            vec3 viewpos0   = screen_viewspace(vec3(coord, scenedepth0));
            vec3 viewpos1   = screen_viewspace(vec3(coord, scenedepth1));

            vec3 scenepos0  = view_scenespace(viewpos0);
            vec3 scenepos1  = view_scenespace(viewpos1);

            vec3 sdir       = normalize(scenepos1);

            float vdotl     = dot(normalize(viewpos1), lightvecView);

            bool water      = mat_id == 102;

            mat2x3 atmos;
                atmos[0] = vec3(0.0);
                atmos[1] = vec3(1.0);

            #ifdef vwater_enabled
                if (water || isEyeInWater == 1) compute_water(atmos, scenepos0, scenepos1, sdir, vdotl);
                #ifdef vfog_translucents
                else compute_fog2(atmos, scenepos0, scenepos1, sdir, !landMask(scenedepth1), vdotl);
                #endif
            #else
                if (!water) compute_fog2(atmos, scenepos0, scenepos1, sdir, !landMask(scenedepth1), vdotl);
            #endif

            tex5.rgb    = atmos[0];
            tex6.rgb    = atmos[1];
        }
    }
    #endif

    /*DRAWBUFFERS:56*/
    gl_FragData[0]  = clampDrawbuffer(tex5);
    gl_FragData[1]  = clampDrawbuffer(tex6);
}