#version 400 compatibility

/*
Copyright (C) 2019 RRe36

All Rights Reserved unless otherwise explicitly stated.


By downloading this you have agreed to the license and terms of use.
These can be found inside the included license-file or here: https://github.com/rre36/glsl_kappa/blob/master/LICENSE

Violating these terms may be penalized with actions according to the Digital Millennium Copyright Act (DMCA), the Information Society Directive and/or similar laws depending on your country.
*/

#include "/lib/common.glsl"
#include "/lib/util/encoders.glsl"

in vec2 coord;

flat in mat4x3 light_color;

flat in vec3 cloud_lightcol;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D colortex5;
uniform sampler2D colortex6;

uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

uniform sampler2D depthtex2;

uniform sampler2D noisetex;

uniform int isEyeInWater;
uniform int frameCounter;
uniform int worldTime;

uniform float eyeAltitude;
uniform float far;
uniform float near;
uniform float frameTimeCounter;
uniform float cloud_bsfade;
uniform float cloud_lflip;
uniform float wetness;

uniform ivec2 eyeBrightness;
uniform ivec2 eyeBrightnessSmooth;

uniform vec2 viewSize;

uniform vec3 cloud_lvec, cloud_lvecView;
uniform vec3 cameraPosition;
uniform vec3 upvecView;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;

vec3 decode3x8(float a){
    int bf = int(a*65535.);
    return vec3(bf%32, (bf>>5)%64, bf>>11) / vec3(31,63,31);
}

vec3 blend_translucencies(vec3 scenecol, vec4 translucents, vec3 albedo) {
    vec3 color  = scenecol;
        color  *= mix(vec3(1.0), albedo, translucents.a);
        color   = mix(color, translucents.rgb, translucents.a);

    return color;
}

vec3 apply_fogdata(vec3 scenecol, mat2x3 data) {
    return scenecol.rgb*data[1] + data[0];
}

float depth_lin(float depth) {
    return (2.0*near) / (far+near-depth * (far-near));
}

vec3 view_screenspace(vec3 viewpos) {
    return ((projMAD(gbufferProjection, viewpos) / -viewpos.z)) *0.5 + 0.5;
}

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

float screen_viewspace(float depth, mat4 projInv) {
    depth   = depth * 2.0 - 1.0;
    return projInv[3].z / (projInv[2].w * depth + projInv[3].w);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

int decodeMatID16(float x) {
    return int(x*65535.0);
}

vec3 simple_fog(vec3 scenecolor, float d, vec3 color) {
    float dist      = max(0.0, d-32.0);
    float density   = 1.0-exp(-dist*3e-3);

    return mix(scenecolor, color, density);
}
vec3 water_fog(vec3 scenecolor, float d, vec3 color) {
    float dist      = max(0.0, d);
    float density   = dist*6.5e-1;
    vec3 scatter   = 1.0-exp(-min(density*0.8, 64e-1)*vec3(0.02, 0.24, 1.0));
    vec3 transmittance = exp(-density*vec3(1.0, 0.28, 0.06)*1.2);

    return scenecolor*transmittance + color*scatter*0.3;
}


#include "/lib/frag/bluenoise.glsl"

/* ------ clouds ------ */

#ifdef vcloud_enabled

    #include "/lib/atmos/project.glsl"

    #define layer2_pass

    #include "/lib/atmos/clouds.glsl"

    vec4 compute_vc(in vec3 vvec, in vec3 wvec, bool terrain, vec3 wpos, inout float layer_transmittance, vec3 skycol) {
        float within    = sstep(eyeAltitude, vcloud_alt-10.0, vcloud_alt) * (1.0-sstep(eyeAltitude, vc_highedge, vc_highedge+10.0));
        bool visible    = eyeAltitude<vcloud_alt ? wvec.y>0.0 : wvec.y<=0.0;

        if (visible || within>0.0) {
            const float vc_midalt = vcloud_alt+vcloud_depth*0.5;

            bool is_below   = eyeAltitude<vc_midalt;

            vec3 bs     = wvec*((vcloud_alt-eyeAltitude)/wvec.y);
            vec3 ts     = wvec*((vc_highedge-eyeAltitude)/wvec.y);

            if (wvec.y<0.0 && is_below || wvec.y>0.0 && !is_below){
                bs = vec3(0.0);
                ts = vec3(0.0);
            }

            vec3 spos   = is_below ? bs : ts;
            vec3 epos   = is_below ? ts : bs;

                spos    = mix(spos, gbufferModelViewInverse[3].xyz, within);
                epos    = mix(epos, wvec*vcloud_clip*0.9, within);

            if (terrain) {
                spos    = gbufferModelViewInverse[3].xyz;
                epos    = wpos;
            }

            float dither = dither_bluenoise();

            #ifdef vcloud_adaptive_samples
            const float bl  = vcloud_depth/vcloud_samples;
            float stepl     = length((epos-spos)/vcloud_samples);
            float stepcoeff = stepl/bl;
                stepcoeff   = 0.45+clamp(stepcoeff-1.1, 0.0, 4.0)*0.5;
                stepcoeff   = mix(stepcoeff, 4.0, sqr(within));
            int steps       = int(vcloud_samples*stepcoeff);
            #else
            float stepcoeff = 1.0 + sqr(within) * 4.0;
            int steps       = int(vcloud_samples*stepcoeff);
            #endif

            vec3 rstep  = (epos-spos)/steps;
            vec3 rpos   = rstep*dither + spos + cameraPosition;
            float rlength = length(rstep);

            vec3 scatter    = vec3(0.0);
            float transmittance = 1.0;
            float fade      = 0.0;
            float fdist     = vcloud_clip+1.0;

            vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloud_lightcol : light_color[3]*0.3;
                sunlight   *= cloud_lflip;
            vec3 skylight   = light_color[1] * (1.0 - sqrt(daytime.w)*0.96);

            float vdotl     = dot(vvec, cloud_lvecView);

            float pfade     = saturate(cloud_mie(vdotl, 0.65));

            #ifdef vcloud_storyMode
            const float sigma_a = 0.1;         //absorption coeff
            const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
            const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
            #else
            const float sigma_a = 0.20;         //absorption coeff
            const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
            const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
            #endif

            for (int i = 0; i<steps; ++i, rpos += rstep) {
                if (transmittance < 0.05) break;
                if (rpos.y < vcloud_alt || rpos.y > vc_highedge) continue;

                float dist  = distance(rpos, cameraPosition);
                float dfade = saturate(dist/vcloud_clip);
                if ((1.0-dfade)<0.01) continue;

                float density   = vcloud_density(rpos);
                if (density<=0.0) continue;

                if (fdist>vcloud_clip) fdist = dist;
                else fdist = mix(fdist, dist, transmittance);

                fade    = linStep(dfade, 0.75, 0.99);

                float extinction = density * sigma_t;
                float stept     = expf(-extinction*rlength);
                float integral  = (1.0 - stept) / sigma_t;

                vec3 result_s   = vec3(0.0);

                #ifdef vcloud_light
                float directod  = vc_directOD(rpos, 6)*sigma_a;
                #else
                float directod  = (1.0-sqr(linStep(rpos.y, vcloud_alt, vc_highedge)))*vcloud_depth*sigma_a*0.75;
                #endif

                float skyod     = sqrt(1.0-linStep(rpos.y, vcloud_alt, vc_highedge)) * vcloud_depth * sigma_a * 0.5;

                float powder    = 1.0 - expf(-density * 25.0);

                #ifdef vcloud_storyMode
                    const float dpowder = 1.0;
                #else
                    float dpowder   = mix(powder, 1.0, pfade);
                #endif

                #ifdef vcloud_fast_ms
                float phase = cloud_phase(vdotl, 1.0);

                result_s.x += max(expf(-directod * sigma_t), expf(-directod * sigma_t * 0.2) * 0.75) * phase * dpowder * sigma_s * 1.25;
                result_s.y += max(expf(-skyod * sigma_t), expf(-skyod * sigma_t * 0.2) * 0.75) * powder * sigma_s * 1.25;
                #else
                for (int j = 0; j<5; ++j) {
                    float n     = float(j);

                    float s_d   = sigma_s * pow(0.42, n);    //scatter derivate
                    float t_d   = sigma_t * pow(0.32, n);    //transmittance/attentuation derivate
                    float phase = cloud_phase(vdotl, pow(0.35, n));  //phase derivate

                    result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
                    result_s.y += expf(-skyod*t_d) * powder * s_d;
                }
                #endif

                scatter    += result_s * integral * transmittance;

                transmittance *= stept;
            }

            fdist   = max(fdist, 0.0);
            transmittance = linStep(transmittance, 0.05, 1.0);

            layer_transmittance *= transmittance;

            scatter.x *= 1.8;
            scatter.y *= 1.3;

            vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

            fade        = saturate(1.0-fade);

            float skyfade = expf(-fdist*cloud_atmos_density);

            //if (fdist < vcloud_clip + 0.5) color = mix(skycol * (1.0 - transmittance), color, skyfade);
            
            transmittance = mix(1.0, transmittance, skyfade*fade);

            return vec4(color*fade*skyfade, transmittance);
        } else {
            return vec4(0.0, 0.0, 0.0, 1.0);
        }
    }

#ifdef vcloud2_enabled
    vec4 compute_vc2(in vec3 vvec, in vec3 wvec, inout float layer_transmittance, vec3 skycol) {
        float within    = sstep(eyeAltitude, vcloud2_alt-10.0, vcloud2_alt) * (1.0-sstep(eyeAltitude, vc2_highedge, vc2_highedge+10.0));
        bool visible    = eyeAltitude<vcloud2_alt ? wvec.y>0.0 : wvec.y<=0.0;

        if (visible || within>0.0) {
            const float vc_midalt = vcloud2_alt+vcloud2_depth*0.5;

            bool is_below   = eyeAltitude<vc_midalt;

            vec3 bs     = wvec*((vcloud2_alt-eyeAltitude)/wvec.y);
            vec3 ts     = wvec*((vc2_highedge-eyeAltitude)/wvec.y);

            if (wvec.y<0.0 && is_below || wvec.y>0.0 && !is_below){
                bs = vec3(0.0);
                ts = vec3(0.0);
            }

            vec3 spos   = is_below ? bs : ts;
            vec3 epos   = is_below ? ts : bs;

                spos    = mix(spos, gbufferModelViewInverse[3].xyz, within);
                epos    = mix(epos, wvec*vcloud2_clip*0.9, within);

            float dither = dither_bluenoise();

            float stepcoeff = 1.0 + sqr(within) * 4.0;
            int steps       = int(vcloud2_samples*stepcoeff);

            vec3 rstep  = (epos-spos)/steps;
            vec3 rpos   = rstep*dither + spos + cameraPosition;
            float rlength = length(rstep);

            vec3 scatter    = vec3(0.0);
            float transmittance = 1.0;
            float fade      = 0.0;
            float fdist     = vcloud2_clip+1.0;

            vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloud_lightcol : light_color[3]*0.3;
                sunlight   *= cloud_lflip;
            vec3 skylight   = light_color[1] * (1.0 - sqrt(daytime.w)*0.96);

            float vdotl     = dot(vvec, cloud_lvecView);

            float pfade     = saturate(cloud_mie(vdotl, 0.65));

            #ifdef vcloud_storyMode
            const float sigma_a = 0.1;         //absorption coeff
            const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
            const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
            #else
            const float sigma_a = 0.20;         //absorption coeff
            const float sigma_s = 1.25;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
            const float sigma_t = 1.25;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
            #endif

            for (int i = 0; i<steps; ++i, rpos += rstep) {
                if (transmittance < 0.05) break;
                if (rpos.y < vcloud2_alt || rpos.y > vc2_highedge) continue;

                float dist  = distance(rpos, cameraPosition);
                float dfade = saturate(dist/vcloud2_clip);
                if ((1.0-dfade)<0.01) continue;

                float density   = vcloud2_density(rpos);
                if (density<=0.0) continue;

                if (fdist>vcloud2_clip) fdist = dist;
                else fdist = mix(fdist, dist, transmittance);

                fade    = linStep(dfade, 0.75, 0.99);

                float extinction = density * sigma_t;
                float stept     = expf(-extinction*rlength);
                float integral  = (1.0 - stept) / sigma_t;

                vec3 result_s   = vec3(0.0);

                #ifdef vcloud_light
                float directod  = vc2_directOD(rpos, 6)*sigma_a;
                #else
                float directod  = (1.0-sqr(linStep(rpos.y, vcloud2_alt, vc2_highedge)))*vcloud2_depth*sigma_a*0.6;
                #endif

                float skyod     = sqrt(1.0-linStep(rpos.y, vcloud2_alt, vc2_highedge)) * vcloud2_depth * sigma_a * 0.25;

                float powder    = 1.0 - expf(-density * 25.0);

                #ifdef vcloud_storyMode
                    const float dpowder = 1.0;
                #else
                    float dpowder   = mix(powder, 1.0, pfade);
                #endif

                #ifdef vcloud_fast_ms
                float phase = cloud_phase(vdotl, 1.0);

                result_s.x += max(expf(-directod * sigma_t), expf(-directod * sigma_t * 0.2) * 0.75) * phase * dpowder * sigma_s * 1.25;
                result_s.y += max(expf(-skyod * sigma_t), expf(-skyod * sigma_t * 0.2) * 0.75) * powder * sigma_s * 1.25;
                #else
                for (int j = 0; j<5; ++j) {
                    float n     = float(j);

                    float s_d   = sigma_s * pow(0.42, n);    //scatter derivate
                    float t_d   = sigma_t * pow(0.32, n);    //transmittance/attentuation derivate
                    float phase = cloud_phase(vdotl, pow(0.35, n));  //phase derivate

                    result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
                    result_s.y += expf(-skyod*t_d) * powder * s_d;
                }
                #endif
                
                scatter    += result_s * integral * transmittance;

                transmittance *= stept;
            }

            fdist   = max(fdist, 0.0);
            transmittance = linStep(transmittance, 0.05, 1.0);
            transmittance = mix(1.0, transmittance, layer_transmittance);
            scatter *= layer_transmittance;
            layer_transmittance *= transmittance;

            scatter.x *= 1.6;
            scatter.y *= 1.0;

            vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

            fade        = saturate(1.0-fade);

            float skyfade = expf(-fdist*cloud_atmos_density*0.9);
            
            //if (fdist < vcloud2_clip + 0.5) color = mix(skycol * (1.0 - transmittance), color, skyfade);
            
            transmittance = mix(1.0, transmittance, fade*skyfade);

            return vec4(color*fade*skyfade, transmittance);
        } else {
            return vec4(0.0, 0.0, 0.0, 1.0);
        }
    }
#endif
#endif

#ifdef reflections_enabled
    //based on robobo1221's shaders because my old ssr is shit
    vec4 ssrTrace(vec3 rdir, vec3 vpos, vec3 screenpos, float dither, float nDotV, const int steps, const int refine) {
        //const int steps     = 16;
        //const int refine    = 5;

        float rlength   = ((vpos.z + rdir.z * far * sqrt(3.0)) > -near) ? (-near - vpos.z) / rdir.z : far * sqrt(3.0);
        vec3 dir        = normalize(view_screenspace(rdir * rlength + vpos) - screenpos);
            dir.xy      = normalize(dir.xy);

        float maxlength = rcp(float(steps));
        float minlength = maxlength*0.05;

        float steplength = mix(minlength, maxlength, (max(nDotV, 0.0)));
        float stepweight = 1.0 / abs(dir.z);

        vec3 pos        = screenpos + dir*steplength*(dither + 0.5);

        float depth     = texelFetch(depthtex1, ivec2(pos.xy * viewSize), 0).x;
        bool ray_hit     = false;

        int i       = steps;
        int j       = refine;

        while (--i > 0) {
            steplength = clamp((depth - pos.z) * stepweight, minlength, maxlength);
            pos += dir*steplength;
            depth = texelFetch(depthtex1, ivec2(pos.xy * viewSize), 0).x;

            if (saturate(pos) != pos) return vec4(0.0);

            if (depth <= pos.z) break;
        }
        float mdepth    = depth;

        vec3 refpos     = pos;
        float refdepth  = depth;

        while (--j > 0) {
            refpos      = dir * clamp((depth - pos.z)*stepweight, -steplength, steplength) + pos;
            refdepth    = texelFetch(depthtex1, ivec2(refpos.xy * viewSize), 0).x;
            bool rayhit = refdepth <= refpos.z;
            if (rayhit) ray_hit = true;

            pos         = rayhit ? refpos : pos;
            depth       = rayhit ? refdepth : depth;

            steplength *= 0.5;
        }

        float sdepth    = texture(depthtex1, pos.xy).x;

        if (sdepth >= 1.0) return vec4(0.0);

        bool visible    = abs(pos.z - mdepth) * min(stepweight, 400.0) <= maxlength;

        return visible ? vec4(texture(colortex0, pos.xy).rgb, 1.0) : vec4(0.0);
    }
    vec4 ssrTrace(vec3 rdir, vec3 vpos, vec3 screenpos, float dither, float nDotV) {
        return ssrTrace(rdir, vpos, screenpos, dither, nDotV, 16, 5);
    }

    float fresnel_schlick(vec3 dir, vec3 normal) {
        vec3 halfdir    = normalize(normal + dir);

        float cosine    = dot(halfdir, dir);
        float fresnel   = max(cosine, 0.0);

        return pow3(fresnel);
    }

    vec2 unproject_sphere(vec3 dir) {
        vec2 lonlat     = vec2(atan(-dir.x, dir.z), acos(dir.y));
        return lonlat * vec2(rcp(tau), rcp(pi)) + vec2(0.5, 0.0);
    }

    vec4 glossy_reflections(vec3 viewpos, vec3 viewdir, vec3 viewnormal, float depth, float nDotV, float dither, float mult) {
        vec3 rdir       = (reflect(viewdir, viewnormal));

        #ifdef ssr_enabled
            vec4 ssr        = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV);
        #else
            const vec4 ssr  = vec4(0.0);
        #endif

        vec3 sdir   = normalize(mat3(gbufferModelViewInverse)*rdir);

            mult   *= saturate(1.0 - dot(-upvecView, viewnormal)) * 0.9 + 0.1;

        vec2 spherecoord = unproject_sphere(sdir);
            spherecoord = spherecoord*rcp(SKYREF_LOD) + vec2(0.55);

        vec3 reflection = texture(colortex5, spherecoord).rgb*mult;
            reflection  = mix(reflection, ssr.rgb, ssr.a);

        return vec4(reflection, max(mult, ssr.a));
    }

    /*
    const vec4 hashscale4   = vec4(443.897, 441.423, 437.195, 444.129);

    vec4 hash42(float i, vec3 viewpos) {
        vec2 p  = viewpos.xy + (frameCounter%8)*0 - i;
        vec4 p4 = fract(p.xyxy * hashscale4);
            p4 += dot(p4, p4.wzxy + 19.19);

        return fract((p4.xxyz + p4.yzzw) * p4.zywx);
    }

    vec3 rough_normal(vec4 noise, float roughness, vec3 normal) {
        noise.xyz   = normalize(cross(normal, noise.xyz));

        return normalize(noise.xyz * (roughness * noise.w * rcp(1.0 - noise.w)) + normal);
    }*/

    /*
    These two functions used for rough reflections are based on zombye's spectrum shaders
    https://github.com/zombye/spectrum
    */

    mat3 get_rotation_mat(vec3 x, vec3 y) {
        float cosine = dot(x, y);
        vec3 axis = cross(y, x);

        float tmp = 1.0 / dot(axis, axis);
            tmp = tmp - tmp * cosine;
        vec3 tmpv = axis * tmp;

        return mat3(
            axis.x * tmpv.x + cosine, axis.x * tmpv.y - axis.z, axis.x * tmpv.z + axis.y,
            axis.y * tmpv.x + axis.z, axis.y * tmpv.y + cosine, axis.y * tmpv.z - axis.x,
            axis.z * tmpv.x - axis.y, axis.z * tmpv.y + axis.x, axis.z * tmpv.z + cosine
        );
    }
    vec3 ggx_facet_dist(vec3 viewdir, float roughness, vec2 xy) {
        /*
        GGX VNDF sampling
        http://www.jcgt.org/published/0007/04/01/
        */

        viewdir     = normalize(vec3(roughness * viewdir.xy, viewdir.z));

        float clsq  = dot(viewdir.xy, viewdir.xy);
        vec3 T1     = vec3(clsq > 0.0 ? vec2(-viewdir.y, viewdir.x) * inversesqrt(clsq) : vec2(1.0, 0.0), 0.0);
        vec3 T2     = vec3(-T1.y * viewdir.z, viewdir.z * T1.x, viewdir.x * T1.y - T1.x * viewdir.y);

        float r     = sqrt(xy.x);
        float phi   = tau * xy.y;
        float t1    = r * cos(phi);
        float a     = saturate(1.0 - t1 * t1);
        float t2    = mix(sqrt(a), r * sin(phi), 0.5 + 0.5 * viewdir.z);

        vec3 normal_h = t1 * T1 + t2 * T2 + sqrt(saturate(a - t2 * t2)) * viewdir;

        return normalize(vec3(roughness * normal_h.xy, normal_h.z));
    }

    vec4 rough_reflections(vec3 viewpos, vec3 viewdir, vec3 viewnormal, float depth, float nDotV, float dither, float mult, float roughness) {
        if (roughness > 0.85) return vec4(0.0);

        float rfade     = 1.0 - linStep(roughness, 0.4, 0.85);
            rfade      *= sqrt(rfade);
        
        int steps = 2 + int(min(roughness * 3.0, 2.0));
        float rstep = 1.0 / float(steps);

        vec3 total      = vec3(0.0);
        float ssr_alpha = 0.0;

        mat3 rot    = get_rotation_mat(vec3(0, 0, 1), viewnormal);
        vec3 tangent_v = viewdir * rot;

        vec2 xy     = vec2(0.0);
        vec3 nrm    = vec3(0.0);
        vec3 rdir   = vec3(0.0);
        vec4 ssr    = vec4(0.0);
        vec3 sdir   = vec3(0.0);
        vec2 spherecoord = vec2(0.0);
        vec3 reflection = vec3(0.0);

        float sum_mult  = 0.0;

        for (int i = 0; i<steps; ++i) {
            //vec3 nrm    = vec3(0.0, 0.0, 1.0);
            //    nrm     = rough_normal(hash42(float(i) * rstep, viewpos), roughness, viewnormal);

                xy     = vec2(fract((i + dither) * sqr(32.0) * phi), (i + dither) * rstep);
                nrm    = rot * ggx_facet_dist(-tangent_v, roughness, xy); 

                rdir   = reflect(viewdir, nrm);

            #ifdef ssr_enabled
                if (roughness >= 0.75) {
                    ssr = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV, 6, 4);
                } else if (roughness >= 0.5) {
                    ssr = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV, 8, 4);
                } else if (roughness >= 0.25) {
                    ssr = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV, 8, 4);
                } else if (roughness >= 0.12) {
                    ssr = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV, 10, 5);
                } else {
                    ssr = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV, 12, 5);
                }
            #endif

            sdir   = normalize(mat3(gbufferModelViewInverse)*rdir);

            float smult = saturate(1.0 - dot(-upvecView, nrm)) * 0.9 + 0.1;
            sum_mult   += smult;

            spherecoord = unproject_sphere(sdir);
            spherecoord = spherecoord*rcp(SKYREF_LOD) + vec2(0.55);

            reflection  = texture(colortex5, spherecoord).rgb * (mult * smult);
            reflection  = mix(reflection, ssr.rgb, ssr.a);

            total  += reflection;
            ssr_alpha += ssr.a;
        }

        mult   *= sum_mult * rstep;
        total  *= rstep;
        ssr_alpha *= rstep;

        return vec4(total, max(mult, ssr_alpha)) * rfade;
    }

    #include "/lib/frag/labPBR.glsl"
    #include "/lib/frag/noise.glsl"
    #include "/lib/frag/wetness.glsl"

    #ifdef labpbr_enabled
    void reflection(inout vec3 scenecol, vec3 viewnormal, vec3 scenenormal, float depth, vec2 lmap, int mat_id, vec2 pbr_tex, bool translucent, vec3 metal_hue) {
        vec4 spectex    = vec4(decode2x8(pbr_tex.x), decode2x8(pbr_tex.y));
        vec4 mat_data   = vec4(1.0, 0.02, 0.0, 0.0);
        float emission  = 0.0;
        bool is_metal   = false;

        #ifdef labpbr_enabled
            decode_lab(spectex, mat_data, emission, is_metal);
        #endif

        if (translucent) mat_data.zw = vec2(0.0);

        bool water      = mat_id == 102;

        vec3 screenpos  = vec3(coord, depth);
        vec3 viewpos    = screen_viewspace(vec3(coord, depth));
        vec3 viewdir    = normalize(viewpos);
        vec3 scenepos   = view_scenespace(viewpos);
        //vec3 scenedir   = normalize(scenepos);

        if (wetness > 0.0) {
            float block_wetness = get_wetness(lmap.y, scenenormal, scenepos + cameraPosition);

            mat_data.x  = mix(mat_data.x, 0.04, block_wetness);
            mat_data.y  = mix(mat_data.y, 0.02, block_wetness);
        }
        
        float nDotV     = saturate(dot(viewnormal, viewdir));

        float dither    = dither_bluenoise();

        float f0        = (water || mat_id == 103) ? 0.05 : mat_data.y;
            if (is_metal) f0 = 0.35;

        float fresnel   = max(fresnel_schlick(viewdir, viewnormal), f0);
            //if (mat_id == 103) fresnel *= 0.35;
            //fresnel     = 1.0;

        float fmult     = sqr(sstep(lmap.y, 0.15, 0.95));

        vec4 reflection = vec4(0.0);

        #ifdef rough_reflections_enabled
            if (mat_data.x < 0.002 || water) reflection  = glossy_reflections(viewpos, viewdir, viewnormal, depth, nDotV, dither, fmult);
            else reflection  = rough_reflections(viewpos, viewdir, viewnormal, depth, nDotV, dither, fmult, mat_data.x);
        #else
            reflection  = glossy_reflections(viewpos, viewdir, viewnormal, depth, nDotV, dither, fmult);
            reflection.a *= sqr(1.0 - linStep(mat_data.x, 0.0, 0.9));
        #endif

        if (is_metal) reflection.rgb *= metal_hue;

        if (is_metal) scenecol = mix(scenecol.rgb, reflection.rgb, reflection.a);
        else scenecol = mix(scenecol.rgb, reflection.rgb, fresnel*reflection.a);
        //scenecol        = compute_ssr(viewnormal, screenpos, viewpos, viewdir, rdir, nDotV, dither).rgb;
    }
    #else
    void reflection(inout vec3 scenecol, in vec3 viewnormal, in vec3 scenenormal, in float depth, vec2 lmap, int mat_id) {
        vec3 screenpos  = vec3(coord, depth);
        vec3 viewpos    = screen_viewspace(vec3(coord, depth));
        vec3 viewdir    = normalize(viewpos);
        vec3 scenepos   = view_scenespace(viewpos);
        //vec3 scenedir   = normalize(scenepos);
            
        float block_wetness = 0.0;

        if (wetness > 0.01) {
            block_wetness = get_wetness(lmap.y, scenenormal, scenepos + cameraPosition) * wetness;
        }
        
        if (!(block_wetness > 0.01 || mat_id == 102 || mat_id == 103)) return;

        float nDotV     = saturate(dot(viewnormal, viewdir));

        float dither    = dither_bluenoise();

        float fresnel   = max(fresnel_schlick(viewdir, viewnormal), 0.02);
            if (mat_id == 103) fresnel *= 0.35;
            if (block_wetness > 0.01 && !(mat_id == 102 || mat_id == 103)) fresnel *= sqrt(block_wetness);
            //fresnel     = 1.0;

        float fmult     = sqr(sstep(lmap.y, 0.15, 0.95));

        vec4 reflection = glossy_reflections(viewpos, viewdir, viewnormal, depth, nDotV, dither, fmult);

        scenecol        = mix(scenecol.rgb, reflection.rgb, fresnel*reflection.a);
        //scenecol        = compute_ssr(viewnormal, screenpos, viewpos, viewdir, rdir, nDotV, dither).rgb;
    }
    #endif
#endif

/* bilateral upscale for fog */
mat2x3 textureBilateralDouble(sampler2D tex1, sampler2D tex2, sampler2D depth, const float lod, float fdepth, const vec2 offset) {
    mat2x3 data = mat2x3(0.0);
    float sum   = 0.0;
    ivec2 posD  = ivec2(coord*viewSize);
    ivec2 posT  = ivec2((coord*rcp(lod)+offset)*viewSize);
    vec3 zmult  = vec3((far*near)*2.0, far+near, far-near);
        fdepth  = depth_lin(fdepth);
    
    for (int i = -1; i<2; i++) {
        for (int j = -1; j<2; j++) {
            ivec2 tcDepth = posD + ivec2(vec2(i, j)*lod);
            float dsample = depth_lin(texelFetch(depth, tcDepth, 0).x);
            float w     = abs(dsample-fdepth)*zmult.x<1.0 ? 1.0 : 1e-5;
            ivec2 ct    = posT + ivec2(i, j);
            data[0]    += texelFetch(tex1, ct, 0).rgb*w;
            data[1]    += texelFetch(tex2, ct, 0).rgb*w;
            sum        += w;
        }
    }
    data *= rcp(sum);

    return data;
}

vec3 blend_clouds(vec4 data, vec3 scenecol, vec3 skycol) {
    vec3 c  = mix(skycol, scenecol, sqr(data.a));

    return c * data.a + data.rgb;
}

void main() {
    vec4 scenecol   = stex(colortex0);  //that macro certainly makes it neater
    vec4 tex1       = stex(colortex1);
    vec4 tex2       = stex(colortex2);
    vec4 tex3       = stex(colortex3);

    vec3 scenenormal = decodeNormal(tex1.xy);
    vec2 scenelmap  = decode2x8(tex1.z);
    vec3 viewnormal = normalize(mat3(gbufferModelView) * scenenormal);

    int mat_id      = decodeMatID16(tex2.x);

    float scenedepth0 = stex(depthtex0).x;
    vec3 viewpos0     = screen_viewspace(vec3(coord, scenedepth0));
    vec3 spos0        = view_scenespace(viewpos0);

    float scenedepth1 = stex(depthtex1).x;
    vec3 viewpos1     = screen_viewspace(vec3(coord, scenedepth1));
    vec3 spos1        = view_scenespace(viewpos1);

    bool translucent = (scenedepth0<scenedepth1);

    bool water      = mat_id == 102;

    vec3 translucent_albedo = sqr(decode3x8(tex2.g));

    float dist0     = distance(spos0, gbufferModelViewInverse[3].xyz);
    float dist1     = distance(spos1, gbufferModelViewInverse[3].xyz);

    float cave_fix  = linStep(eyeBrightnessSmooth.y/240.0, 0.1, 0.9);

    #ifdef vcloud_enabled
        float layer_transmittance = 1.0;

        vec3 skycol     = texture(colortex5, projectSky(normalize(spos1))).rgb * 0.95 + scenecol.rgb * 0.05;

        vec4 vcloud     = compute_vc(normalize(viewpos1), normalize(spos1), landMask(scenedepth1), spos1, layer_transmittance, skycol);
        bool vcloud_tblend = translucent && (eyeAltitude < vcloud_alt ? (spos0.y + cameraPosition.y)<vcloud_alt : (spos0.y + cameraPosition.y) > vc_highedge);
            vcloud_tblend = vcloud_tblend && !(eyeAltitude > vcloud_alt && eyeAltitude < vc_highedge);

        #ifdef vcloud2_enabled
        if (!landMask(scenedepth1) && layer_transmittance >= 0.01) {
            vec4 vcloud2   = compute_vc2(normalize(viewpos1), normalize(spos1), layer_transmittance, skycol);
            //scenecol.rgb = scenecol.rgb*vcloud2.a + vcloud2.rgb;
            scenecol.rgb    = blend_clouds(vcloud2, scenecol.rgb, skycol);
                skycol.rgb  = skycol.rgb*vcloud2.a + vcloud2.rgb;
        }
        #endif

    #else
    const vec4 vcloud = vec4(0.0, 0.0, 0.0, 1.0);
    const bool vcloud_tblend = false;
    #endif
    /*
    if (translucent && isEyeInWater==0){
        if (water) scenecol.rgb = water_fog(scenecol.rgb, dist1-dist0, light_color[1]*cave_fix);
        else if (landMask(scenedepth1)) scenecol.rgb = simple_fog(scenecol.rgb, dist1-dist0, light_color[1]*cave_fix);
    }*/

    //if (landMask(scenedepth1) && isEyeInWater==1 && translucent) scenecol.rgb = simple_fog(scenecol.rgb, dist1-dist0, light_color[1]*cave_fix);

    #if (defined vwater_enabled || defined vfog_translucents)
        mat2x3 atmos1;
            atmos1[0] = vec3(0.0);
            atmos1[1] = vec3(1.0);

        if (translucent || isEyeInWater == 1) {
            vec2 fcoord_atmos = saturate(coord*rcp(2.02) + vec2(0.0, 0.5));

            #ifdef vwater_bilateral
                atmos1      = textureBilateralDouble(colortex5, colortex6, depthtex0, 2.02, scenedepth0, vec2(0.0, 0.5));
            #else
                atmos1[0]   = texture(colortex5, fcoord_atmos).rgb;
                atmos1[1]   = texture(colortex6, fcoord_atmos).rgb;
            #endif
        }
        
        if (translucent && isEyeInWater == 0) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos1);
    #endif

    #ifdef vfog_enabled
        vec2 fcoord_atmos = saturate(coord*0.5 + vec2(0.5, 0.0));

        mat2x3 atmos;
            atmos[0] = vec3(0.0);
            atmos[1] = vec3(1.0);

        #ifdef vfog_bilateral
            atmos       = textureBilateralDouble(colortex5, colortex6, depthtex0, 2, scenedepth0, vec2(0.5, 0.0));
        #else
            atmos[0]      = texture(colortex5, fcoord_atmos).rgb;
            atmos[1]      = texture(colortex6, fcoord_atmos).rgb;
        #endif

        if (isEyeInWater == 1) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos);
    #endif

    #ifdef vcloud_enabled
    if (vcloud_tblend) {
        scenecol.rgb = scenecol.rgb*vcloud.a + vcloud.rgb;
        //scenecol.rgb    = blend_clouds(vcloud, scenecol.rgb, skycol);
    }
    #endif

    scenecol.rgb    = blend_translucencies(scenecol.rgb, tex3, translucent_albedo);

    #ifdef reflections_enabled
        #ifdef labpbr_enabled
        if (landMask(scenedepth0)) {
            vec3 metal_hue = (decode3x8(tex2.g));
            reflection(scenecol.rgb, viewnormal, scenenormal, scenedepth0, scenelmap, mat_id, vec2(tex2.z, tex1.w), translucent, metal_hue);
        }
        #else
        if (water || mat_id == 103 || (wetness > 0.01 && landMask(scenedepth0))) reflection(scenecol.rgb, viewnormal, scenenormal, scenedepth0, scenelmap, mat_id);
        #endif
    #endif

    /*
    if (landMask(scenedepth0) && isEyeInWater==0) scenecol.rgb = simple_fog(scenecol.rgb, dist0, light_color[1]*cave_fix);

    if (isEyeInWater==1) scenecol.rgb = water_fog(scenecol.rgb, dist0, light_color[1]*cave_fix);
    */

    #ifdef vcloud_enabled
    if (!vcloud_tblend) {
        //scenecol.rgb = scenecol.rgb*vcloud.a + vcloud.rgb;
        scenecol.rgb    = blend_clouds(vcloud, scenecol.rgb, skycol);
    }
    #endif

    #ifdef vfog_enabled
        if (isEyeInWater == 0) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos);
    #endif

    #if (defined vwater_enabled || defined vfog_translucents)
        if (isEyeInWater == 1) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos1);
    #endif

    if (mat_id == 3) {
        //if (landMask(scenedepth0)) scenecol.rgb = scenecol.rgb*0.6 + vec3(0.5)*v3avg(scenecol.rgb);
        scenecol.rgb = scenecol.rgb*0.7 + vec3(0.9, 0.8, 0.7)*v3avg(scenecol.rgb);
    }

    vec4 return1    = vec4(1.0);

    #ifdef lens_flare
        if (landMask(scenedepth1)) return1.rgb = vec3(0.0);
        else if (translucent) return1.rgb = translucent_albedo;

        return1.rgb *= vcloud.a;
    #endif

    //scenecol.rgb    = stex(colortex5).rgb;

    /*DRAWBUFFERS:01*/
    gl_FragData[0]  = makeDrawbuffer(scenecol);
    gl_FragData[1]  = clampDrawbuffer(return1);
}