#version 400 compatibility

/*
Copyright (C) 2019 RRe36

All Rights Reserved unless otherwise explicitly stated.


By downloading this you have agreed to the license and terms of use.
These can be found inside the included license-file or here: https://github.com/rre36/glsl_kappa/blob/master/LICENSE

Violating these terms may be penalized with actions according to the Digital Millennium Copyright Act (DMCA), the Information Society Directive and/or similar laws depending on your country.
*/

#include "/lib/common.glsl"
#include "/lib/util/encoders.glsl"
#include "/lib/util/srgb.glsl"

const int shadowMapResolution   = 2560;     //[512 1024 1536 2048 2560 3072 3584 4096 6144 8192 16384]
const float shadowDistance      = 128.0;

const bool shadowHardwareFiltering = true;

in vec2 coord;

flat in float light_flip;

flat in mat4x3 light_color;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D colortex3;

uniform sampler2D noisetex;

uniform sampler2D depthtex1;

uniform sampler2DShadow shadowtex0;
uniform sampler2DShadow shadowtex1;
uniform sampler2D shadowcolor0;

uniform int frameCounter;

uniform float aspectRatio;
uniform float far, near;
uniform float sunAngle;
uniform float viewHeight, viewWidth;
uniform float wetness;

uniform vec2 taaOffset;
uniform vec2 viewSize;

uniform vec3 lightvec, lightvecView;
uniform vec3 cameraPosition;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;
uniform mat4 shadowModelView, shadowModelViewInverse;
uniform mat4 shadowProjection, shadowProjectionInverse;

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    screenpos.xy -= taaOffset;

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

float depth_lin(float depth) {
    return (2.0*near) / (far+near-depth * (far-near));
}

#include "/lib/light/diffuse.glsl"

#include "/lib/light/warp.glsl"

#include "/lib/frag/bluenoise.glsl"

#define g_solid

#include "/lib/light/shadow.glsl"

vec3 get_lblock(vec3 lcol, float lmap) {
    lmap    = pow5(lmap);
    lcol    = mix(vec3(lcol.x, lcol.y*0.6, lcol.z*0.8 + lcol.x*0.2), lcol, sqrt(lmap));
    return lmap*lcol;
}

float get_specGGX(vec3 normal, vec3 svec, vec2 material) {
    float f0  = material.y;
    float roughness = sqr(material.x);

    vec3 h      = lightvecView - svec;
    float hn    = inversesqrt(dot(h, h));
    float hDotL = saturate(dot(h, lightvecView)*hn);
    float hDotN = saturate(dot(h, normal)*hn);
    float nDotL = saturate(dot(normal, lightvecView));
    float denom = (hDotN * roughness - hDotN) * hDotN + 1.0;
    float D     = roughness / (pi * denom * denom);
    float F     = f0 + (1.0-f0) * exp2((-5.55473*hDotL-6.98316)*hDotL);
    float k2    = 0.25 * roughness;

    return nDotL * D * F / (hDotL * hDotL * (1.0-k2) + k2);
}

vec3 get_light(vec3 scenecol, vec3 normal, vec3 viewpos, vec2 lmap, float ao, int matID, vec4 mat_data, bool is_metal) {
    float shadow    = 1.0;
    vec3 shadowcol  = vec3(1.0);
    lmap.y          = pow3(lmap.y);

    float diff      = get_diffLambert(normal);
        diff        = matID != 2 ? diff : diff*0.3+0.7;

    get_ldirect(shadow, shadowcol, diff>0.0, viewpos);

    float diff_lit  = min(diff, shadow);
    vec3 direct_col     = sunAngle<0.5 ? light_color[0] : light_color[3];
    vec3 direct_light   = diff_lit*shadowcol*direct_col*light_flip;
        direct_light   *= ao * 0.9 + 0.1;

    vec3 indirect_light = lmap.y*light_color[1];
        indirect_light += vec3(0.5, 0.7, 1.0)*0.01*minlight_luma;
        indirect_light *= ao;

    vec3 result     = direct_light + indirect_light;
        result     += get_lblock(light_color[2], lmap.x)*(ao * sqrt(ao));

    #ifdef labpbr_enabled
    vec3 specular   = direct_col * get_specGGX(mat3(gbufferModelView) * normal, normalize(viewpos), mat_data.xy);
        specular   *= diff_lit * light_flip;
    if (is_metal) {
        specular   *= scenecol.rgb;
        scenecol.rgb *= scenecol.rgb;
    }

    return scenecol * result + specular;
    
    #else

    return scenecol * result;

    #endif
}

int decodeMatID16(float x) {
    return int(x*65535.0);
}

vec4 textureBilateral(sampler2D tex, sampler2D depth, const int lod, float fdepth) {
    vec4 data   = vec4(0.0);
    float sum   = 0.0;
    ivec2 posD  = ivec2(coord*viewSize);
    ivec2 posT  = ivec2(coord*viewSize*rcp(float(lod)));
    vec3 zmult  = vec3((far*near)*2.0, far+near, far-near);
        fdepth  = depth_lin(fdepth);
    
    for (int i = -1; i<2; i++) {
        for (int j = -1; j<2; j++) {
            ivec2 tcDepth = posD + ivec2(i, j)*lod;
            float dsample = depth_lin(texelFetch(depth, tcDepth, 0).x);
            float w     = abs(dsample-fdepth)*zmult.x<1.0 ? 1.0 : 1e-5;
            ivec2 ct    = posT + ivec2(i, j);
            data       += texelFetch(tex, ct, 0)*w;
            sum        += w;
        }
    }
    data *= rcp(sum);

    return data;
}

float bayer2e(vec2 a){
    a = floor(a);
    return fract( dot(a, vec2(.5, a.y * .75)) );
}
#define bayer4e(a)   (bayer2e( .5*(a))*.25+bayer2e(a))

#define m vec3(31,63,31)
float encode3x8(vec3 a){
    float dither = bayer4e(gl_FragCoord.xy);
    a += (dither-.5) / m;
    a = saturate(a);
    ivec3 b = ivec3(a*m);
    return float( b.r|(b.g<<5)|(b.b<<11) ) / 65535.;
}
#undef m

#include "/lib/frag/labPBR.glsl"
#include "/lib/frag/noise.glsl"
#include "/lib/frag/wetness.glsl"

void main() {
    vec4 scenecol   = stex(colortex0);
    float scenedepth = stex(depthtex1).x;

    vec4 tex2       = stex(colortex2);
    vec4 return2    = tex2;

    if (landMask(scenedepth)) {
        vec4 tex1       = stex(colortex1);
        vec3 scenenormal = decodeNormal(tex1.xy);
        vec2 scenelmap  = decode2x8(tex1.z);

        int matID       = decodeMatID16(tex2.x);

        #if DEBUG_VIEW==1
        scenecol.rgb    = vec3(1.0);
        #endif

        vec3 viewpos    = screen_viewspace(vec3(coord, scenedepth));
        vec3 scenepos   = view_scenespace(viewpos);

        float ao        = pow3(scenecol.a)*0.94+0.06;

        #ifdef ambientOcclusion_enabled
            vec4 tex3   = textureBilateral(colortex3, depthtex1, 2, scenedepth);
                ao     *= tex3.r;
        #endif

        vec4 spectex    = vec4(decode2x8(tex2.z), decode2x8(tex1.w));
        vec4 mat_data   = vec4(1.0, 0.02, 0.6, 0.0);
        float emission  = 0.0;
        bool is_metal   = false;

        float block_wetness = 0.0;

        if (wetness > 0.01) block_wetness = get_wetness(scenelmap.y, scenenormal, scenepos + cameraPosition) * wetness;

        #ifdef labpbr_enabled
        decode_lab(spectex, mat_data, emission, is_metal);
        scenecol.rgb    = compute_porosity(scenecol.rgb, vec3(mat_data.xz, block_wetness));
        if (block_wetness > 0.01) {
            mat_data.x  = mix(mat_data.x, 0.04, block_wetness);
            mat_data.y  = mix(mat_data.y, 0.02, block_wetness);
        }
        vec3 emitted    = scenecol.rgb * emission;
        #else
        scenecol.rgb    = compute_porosity(scenecol.rgb, vec3(mat_data.xz, block_wetness));
        #endif

        if (is_metal) return2.y = encode3x8(normalize(scenecol.rgb) * (v3avg(scenecol.rgb) * 0.5 + 0.5));

        scenecol.rgb    = get_light(scenecol.rgb, scenenormal, viewpos, scenelmap, ao, matID, mat_data, is_metal);

        #if DEBUG_VIEW==2
        scenecol.rgb    = ao * (sunAngle<0.5 ? light_color[0] : light_color[3]);
        #endif

        #ifdef labpbr_enabled
        scenecol.rgb   += emitted * pi;
        #endif
    }

    /*DRAWBUFFERS:023*/
    gl_FragData[0]  = makeDrawbuffer(scenecol);
    gl_FragData[1]  = clampDrawbuffer(return2);
    gl_FragData[2]  = vec4(0.0);
}