#version 400 compatibility

/*
====================================================================================================

    Copyright (C) 2020 RRe36

    All Rights Reserved unless otherwise explicitly stated.


    By downloading this you have agreed to the license and terms of use.
    These can be found inside the included license-file
    or here: https://rre36.github.io/license/

    Violating these terms may be penalized with actions according to the Digital Millennium
    Copyright Act (DMCA), the Information Society Directive and/or similar laws
    depending on your country.

====================================================================================================
*/

#include "/lib/common.glsl"

const int noiseTextureResolution = 256;

in vec2 coord;

flat in float light_flip;

flat in mat4x3 light_color;

uniform sampler2D colortex0;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D colortex5;
uniform sampler2D colortex6;

uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform sampler2DShadow shadowtex0;
uniform sampler2DShadow shadowtex1;
uniform sampler2D shadowcolor0;

uniform int frameCounter;
uniform int isEyeInWater;

uniform float eyeAltitude;
uniform float frameTimeCounter;
uniform float sunAngle;
uniform float wetness;

uniform ivec2 eyeBrightness;
uniform ivec2 eyeBrightnessSmooth;

uniform vec2 pixelSize, viewSize;
uniform vec2 taaOffset;

uniform vec3 cameraPosition;
uniform vec3 lightvec, lightvecView;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;
uniform mat4 shadowModelView, shadowModelViewInverse;
uniform mat4 shadowProjection, shadowProjectionInverse;

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    #ifdef taa_enabled
        screenpos.xy -= taaOffset;
    #endif

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

int decodeMatID16(float x) {
    return int(x*65535.0);
}

#include "/lib/light/warp.glsl"

vec3 get_shadowcoord(vec3 viewpos) {  //shadow 2d
    vec3 pos    = viewpos;
        pos    += vec3(0.08)*lightvec;
        pos     = viewMAD(shadowModelView, pos);
        pos     = projMAD(shadowProjection, pos);
        pos.z  *= 0.2;

        pos.xy  = warp_shadowmap(pos.xy);

    return pos*0.5+0.5;
}

vec3 get_shadowcol(sampler2D tex, vec2 coord) {
    vec4 x  = texture(tex, coord);
    return mix(vec3(1.0), x.rgb, x.a);
}

#include "/lib/frag/bluenoise.glsl"

#include "/lib/frag/noise.glsl"

const float fog_alt = 90.0;
const float fog_maxalt = fog_alt+70.0;
const float fog_clip = 448.0;
//float fog_density = 0.5 * (0.50*daytime.x + 0.24*daytime.y + 0.25*daytime.z + 0.40*daytime.w);
//float fog_mist_density = max(daytime.x + daytime.z * 0.05 + daytime.w * 0.5, wetness * 1.2);
float fog_tick  = frameTimeCounter*0.2;

uniform vec3 fog_density;

#include "/lib/atmos/phase.glsl"

#include "/lib/atmos/aconst.glsl"

vec2 phaseFunction(float cosTheta) { 
    return vec2(rayleigh_phase(cosTheta), hg_mie(cosTheta, mie_g*0.95));
}

float fog_mist_phase(float cos_theta) {
    float a     = cs_mie(cos_theta, 0.72);
    float b     = hg_mie(cos_theta, -0.22);

    return mix(a, b, 0.2) + 0.01;
}

vec3 get_fog_density(vec3 pos) {
    float fade  = 1.0 - sstep(pos.y, fog_maxalt - 20.0, fog_maxalt);
    float mistfade = sstep(pos.y, fog_alt - 25.0, fog_alt);
    vec3 wind   = vec3(fog_tick, fog_tick*0.1, fog_tick*0.03);

    float rm    = expf(-max(pos.y - fog_alt + 26.0, 0.0) * 14e-3) * fade;

    pos    += wind;
    pos    *= 0.04;

    float shape = 0.0;

    #ifdef vfog_mist_enabled
        if (mistfade < 1.0 && fog_density.z > 1e-6) {
            float md = sstep(value_3d(pos), 0.36, 0.9);
            if (md < 0.99) {
            shape  += value_3d(pos * 8.4 + shape * 0.2) * 1.0;
            #ifdef vfog_hq_mist
            shape  += value_3d(pos * 21.4 + shape * 0.6) * 0.8;
            //shape  += value_3d(pos * 38.4 + shape * 1.1) * 0.5;
            shape  /= 1.0 + 0.8;
            #endif
            shape  -= 0.25 + md + mistfade * 0.2;
            shape  *= (1.0 - md);
            shape  *= saturate(2.0 - mistfade * 2.0);
            shape   = max(shape, 0.0);
            }
        }
    #endif

    vec3 ret    = vec3(vec2(rm) * 0.25, shape) * fog_density;
        ret.y   = max(ret.y, wetness * 0.2 * fade);

    return ret*vfog_density;
}

#ifdef vfog_bilateral
    #define vf_bstep int(12*vfog_quality)
    #define vf_dstep 14
#else
    #define vf_bstep int(16*vfog_quality)
    #define vf_dstep 16
#endif

void compute_fog(out mat2x3 data, vec3 scenepos0, vec3 scenepos1, vec3 svec, bool sky, float vdotl) {
    //const int steps = 28;
    const float air_density = 1e3;

    float ts        = length(svec*((fog_maxalt-eyeAltitude)/svec.y));
    float bs        = length(svec*((-32.0-eyeAltitude)/svec.y));

    float sd        = svec.y>0.0 ? ts : bs;

    float edist     = sky ? min(sd, fog_clip) : length(scenepos0);
    float sdist     = eyeAltitude>fog_maxalt ? ts : 1.0;

    vec3 spos       = eyeAltitude>fog_maxalt ? sdist*svec + gbufferModelViewInverse[3].xyz : gbufferModelViewInverse[3].xyz;
    vec3 epos       = sky ? edist*svec : scenepos0;

    if (isEyeInWater == 1) {
        spos        = scenepos0;
        epos        = sky ? epos : scenepos1;
    }

    float dither    = dither_bluenoise();

    float bstep      = length((epos-spos));
    float stepcoeff  = saturate(bstep/256.0);

    int steps        = vf_bstep+int(stepcoeff*vf_dstep);

    vec3 rstep      = (epos-spos)/steps;
    vec3 rpos       = spos + rstep*dither;
    float rlength   = length(rstep);

    mat2x3 scatter  = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    vec3 phase      = vec3(phaseFunction(vdotl), fog_mist_phase(vdotl));
    float phaseIso  = 0.33*rcp(pi);

    float cave_fix  = lin_step(eyeBrightness.y/240.0, 0.1, 0.9);

    vec3 sunlight   = (sunAngle<0.5 ? light_color[0] : light_color[3])*light_flip*pi;
    vec3 skylight   = light_color[1]*cave_fix;

    for (int i = 0; i<steps; ++i, rpos += rstep) {
        if (max3(transmittance) < 0.01) break;
        if ((rpos.y+cameraPosition.y)>fog_maxalt) continue;
        vec3 density   = get_fog_density(rpos+cameraPosition);
        if (v3avg(density) <= 0.0) continue;

        vec3 airmass = density * rlength * air_density;
        vec3 extinction = fog_extinct_mat * airmass;
    
        vec3 stept  = expf(-extinction);
        vec3 integral = saturate((stept - 1.0) / -max(extinction, 1e-16));
        vec3 vs     = transmittance * integral;

        vec3 sun_s  = fog_scatter_mat * (airmass * phase) * vs;
        vec3 sky_s  = fog_scatter_mat * (airmass * phaseIso) * vs;

        vec3 shadowpos = get_shadowcoord(rpos);

        float s0        = shadow2D(shadowtex0, shadowpos).x;
        float shadow    = 1.0;
        vec3 shadowcol  = vec3(1.0);

        if (s0<1.0) {
            shadow      = shadow2D(shadowtex1, shadowpos).x;

            if (distance(shadow, s0)>0.1) {
                shadowcol = get_shadowcol(shadowcolor0, shadowpos.xy);
            } else {
                shadowcol = vec3(1.0);
            }
        }

        scatter[0]      += sun_s * shadow * transmittance * shadowcol;
        scatter[1]      += sky_s * transmittance;
        transmittance  *= stept;
    }

    vec3 color      = scatter[0]*sunlight + scatter[1]*skylight;

    data[0]         = color;
    data[1]         = transmittance;
}

void compute_water(out mat2x3 data, vec3 scenepos0, vec3 scenepos1, vec3 svec, float vdotl) {
    const float water_density = 1.2 * vwater_density;

    bool in_water   = isEyeInWater == 1;

    vec3 epos       = in_water ? scenepos0 : scenepos1;
    vec3 spos       = in_water ? gbufferModelViewInverse[3].xyz : scenepos0;

    float bstep      = length((epos-spos));
    float stepcoeff  = saturate(bstep/64.0);
    int steps        = 6;

    if (isEyeInWater == 1) steps = 8+int(stepcoeff*8*vwater_quality);
    else steps = 4+int(stepcoeff*6*vwater_quality);

    float dither    = dither_bluenoise();

    vec3 rstep      = (epos-spos)/steps;
    vec3 rpos       = spos + rstep*dither;
    float rlength   = length(rstep);

    mat2x3 scatter  = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    float phase     = hg_mie(vdotl, 0.76);
    float phaseIso  = 0.25*rcp(pi);
        phase       = phaseIso*0.2 + phase*0.8;

    float cave_fix  = lin_step(eyeBrightnessSmooth.y/240.0, 0.1, 0.9);

    vec3 sunlight   = (sunAngle<0.5 ? light_color[0]*pi : light_color[3])*light_flip;
    vec3 skylight   = light_color[1]*(cave_fix);

    const vec3 extinct_coeff = vec3(1e0, 5e-1, 3e-1) * 5e-1;
    const vec3 scatter_coeff = vec3(5e-2, 6e-2, 8e-2);

    for (int i = 0; i<steps; ++i, rpos += rstep) {
        if (max3(transmittance) < 0.01) break;

        float watermass = water_density * rlength;
        vec3 od     = extinct_coeff * watermass;

        vec3 stept  = saturate(expf(-od));
        vec3 integ  = saturate((stept - 1.0) / -od);
        vec3 vs     = transmittance * integ;

        vec3 sun_s  = scatter_coeff * watermass * phase * vs;
        vec3 sky_s  = scatter_coeff * watermass * phaseIso * vs;

        vec3 shadowpos = get_shadowcoord(rpos);

        float shadow    = shadow2D(shadowtex1, shadowpos).x;

        #ifdef vwater_caustic
        if (shadow > 0.01) {
            vec4 shadowcol  = texture(shadowcolor0, shadowpos.xy);

            if (shadowcol.a > 0.9 && shadowcol.a < 0.99) {
                shadow *= saturate((shadowcol.b - 0.36)*2.0);
            }
        }
        #endif

        scatter[0]      += sun_s * shadow * transmittance;
        scatter[1]      += sky_s * transmittance;
        transmittance  *= stept;
    }
    transmittance   = lin_step(transmittance, 0.01, 1.0);

    vec3 color      = scatter[0]*sunlight + scatter[1]*skylight;

    data[0]         = color;
    data[1]         = transmittance;
}

void compute_fog2(inout mat2x3 data, vec3 scenepos0, vec3 scenepos1, vec3 svec, bool sky, float vdotl) {
    //const int steps = 8;
    const float air_density = 1e3;

    float ts        = length(svec*((fog_maxalt-eyeAltitude)/svec.y));
    float bs        = length(svec*((-32.0-eyeAltitude)/svec.y));

    float sd        = svec.y>0.0 ? ts : bs;

    float sdist     = eyeAltitude>fog_maxalt ? ts : length(scenepos1);
    float edist     = sky ? min(sd, fog_clip) : length(scenepos0);

    vec3 spos       = eyeAltitude>fog_maxalt ? sdist*svec : scenepos0;
    vec3 epos       = sky ? edist*svec : scenepos1;

    float dither    = dither_bluenoise();

    float bstep      = length((epos-spos));
    float stepcoeff  = saturate(bstep/256.0);
    int steps        = 6+int(stepcoeff*8);

    vec3 rstep      = (epos-spos)/steps;
    vec3 rpos       = spos + rstep*dither;
    float rlength   = length(rstep);

    mat2x3 scatter  = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    vec3 phase      = vec3(phaseFunction(vdotl), fog_mist_phase(vdotl));
    float phaseIso  = 0.33*rcp(pi);

    float cave_fix  = lin_step(eyeBrightness.y/16.0, 0.1, 0.9);

    vec3 sunlight   = (sunAngle<0.5 ? light_color[0] : light_color[3])*light_flip*pi;
    vec3 skylight   = light_color[1]*cave_fix;

    for (int i = 0; i<steps; ++i, rpos += rstep) {
        if (max3(transmittance) < 0.01) break;
        if ((rpos.y+cameraPosition.y)>fog_maxalt) continue;
        vec3 density   = get_fog_density(rpos+cameraPosition);
        if (v3avg(density) <= 0.0) continue;

        vec3 airmass = density * rlength * air_density;
        vec3 extinction = fog_extinct_mat * airmass;
    
        vec3 stept  = expf(-extinction);
        vec3 integral = saturate((stept - 1.0) / -max(extinction, 1e-16));
        vec3 vs     = transmittance * integral;

        vec3 sun_s  = fog_scatter_mat * (airmass * phase) * vs;
        vec3 sky_s  = fog_scatter_mat * (airmass * phaseIso) * vs;

        vec3 shadowpos = get_shadowcoord(rpos);

        float s0        = shadow2D(shadowtex0, shadowpos).x;
        float shadow    = 1.0;
        vec3 shadowcol  = vec3(1.0);

        if (s0<1.0) {
            shadow      = shadow2D(shadowtex1, shadowpos).x;

            if (distance(shadow, s0)>0.1) {
                shadowcol = get_shadowcol(shadowcolor0, shadowpos.xy);
            } else {
                shadowcol = vec3(1.0);
            }
        }

        scatter[0]      += sun_s * shadow * transmittance * shadowcol;
        scatter[1]      += sky_s * transmittance;
        transmittance  *= stept;
    }

    vec3 color      = scatter[0]*sunlight + scatter[1]*skylight;

    data[0]        += color;
    data[1]        *= transmittance;
}

vec3 apply_fogdata(vec3 scenecol, mat2x3 data) {
    return scenecol.rgb*data[1] + data[0];
}

float min_depth3x3(sampler2D depthtex, vec2 coord, vec2 px) {

    float tl    = texture(depthtex, coord + vec2(-px.x, -px.y)).x;
    float tc    = texture(depthtex, coord + vec2(0.0, -px.y)).x;
    float tr    = texture(depthtex, coord + vec2(px.x, -px.y)).x;
    float tmin  = min(tl, min(tc, tr));

    float ml    = texture(depthtex, coord + vec2(-px.x, 0.0)).x;
    float mc    = texture(depthtex, coord).x;
    float mr    = texture(depthtex, coord + vec2(px.x, 0.0)).x;
    float mmin  = min(ml, min(mc, mr));

    float bl    = texture(depthtex, coord + vec2(-px.x, px.y)).x;
    float bc    = texture(depthtex, coord + vec2(0.0, px.y)).x;
    float br    = texture(depthtex, coord + vec2(px.x, px.y)).x;
    float bmin  = min(bl, min(bc, br));

    return min(tmin, min(mmin, bmin));
}

#include "/lib/atmos/project.glsl"

vec3 apply_clouds(mat2x4 data, vec3 scenecol, vec3 skycol) {
    scenecol    = mix(skycol, scenecol, data[0].a);
    return mix(scenecol, scenecol * data[0].a + data[0].rgb, data[1].rgb);
}
vec3 apply_clouds(mat2x4 data, vec3 scenecol) {
    return mix(scenecol, scenecol * data[0].a + data[0].rgb, data[1].rgb);
}

const float vcloud_maxalt   = vcloud_alt + vcloud_depth;

vec4 texturePixel(sampler2D tex, vec2 coord, const int lod) {
    return texelFetch(tex, ivec2(coord * viewSize * lod), 0);
}
vec4 texturePixel(sampler2D tex, vec2 coord) {
    return texelFetch(tex, ivec2(coord * viewSize), 0);
}

void main() {
    vec4 scenecol   = stex(colortex0);  //that macro certainly makes it neater
    //vec4 tex3       = stex(colortex3);
    vec4 tex5       = stex(colortex5);
    vec4 tex6       = vec4(0.0, 0.0, 0.0, 1.0);

    //tex5.rgb    = vec3(0.0);
    tex6.rgb    = vec3(1.0);

    #ifdef vcloud_enabled
        const float cLOD    = sqrt(CLOUD_RENDER_LOD);
        vec2 cloudcoord = (coord)*rcp(cLOD)+vec2(1.0-rcp(cLOD), 0.0);

        mat2x4 cloud_data   = mat2x4(texture(colortex6, cloudcoord), texture(colortex5, cloudcoord));

        float scenedepth1 = stex(depthtex1).r;
        vec3 viewpos1   = screen_viewspace(vec3(coord, scenedepth1));
        vec3 scenepos1  = view_scenespace(viewpos1);

        bool is_cloud   = ((scenepos1.y + eyeAltitude) > vcloud_alt && eyeAltitude < vcloud_alt) ||
                        ((scenepos1.y + eyeAltitude) < vcloud_maxalt && eyeAltitude > vcloud_maxalt);
            is_cloud    = is_cloud || (eyeAltitude > vcloud_alt && eyeAltitude < vcloud_maxalt);

        if (!landMask(scenedepth1)) {
            vec3 skycol     = texture(colortex5, projectSky(normalize(scenepos1))).rgb;
            scenecol.rgb = apply_clouds(cloud_data, scenecol.rgb, skycol);
        } else if (is_cloud) {
            scenecol.rgb = apply_clouds(cloud_data, scenecol.rgb);
        }
    #endif

    #ifdef vfog_enabled
    //atmosfog
    vec2 fogcoord   = (coord - vec2(0.5, 0.0))*2.0;
    if (fogcoord == clamp(fogcoord, -0.003, 1.003)) {
        vec2 coord      = saturate(fogcoord);

        float scenedepth0 = texturePixel(depthtex0, coord).r;
        float scenedepth1 = texturePixel(depthtex1, coord).r;

        vec3 viewpos0   = screen_viewspace(vec3(coord, scenedepth0));
        vec3 viewpos1   = screen_viewspace(vec3(coord, scenedepth1));

        vec3 scenepos0  = view_scenespace(viewpos0);
        vec3 scenepos1  = view_scenespace(viewpos1);

        vec3 sdir       = normalize(scenepos0);

        float vdotl     = dot(normalize(viewpos0), lightvecView);

        mat2x3 atmos;
            atmos[0] = vec3(0.0);
            atmos[1] = vec3(1.0);

        compute_fog(atmos, scenepos0, scenepos1, sdir, !landMask(scenedepth0), vdotl);
        tex5.rgb    = atmos[0];
        tex6.rgb    = atmos[1];
    }
    #endif

    #if (defined vwater_enabled || defined vfog_translucents)
    //water volume
    vec2 fogcoordw  = (coord - vec2(0.0, 0.5))*2.02;
    if (fogcoordw == clamp(fogcoordw, -0.003, 1.003)) {
        vec2 coord      = saturate(fogcoordw);
        vec4 tex2       = texturePixel(colortex2, coord);
        int mat_id      = decodeMatID16(tex2.x);

        float scenedepth0 = min_depth3x3(depthtex0, coord, pixelSize);
        float scenedepth1 = texturePixel(depthtex1, coord).r;

        bool translucent = (scenedepth0<scenedepth1);

        if (translucent || isEyeInWater==1) {
            vec3 viewpos0   = screen_viewspace(vec3(coord, scenedepth0));
            vec3 viewpos1   = screen_viewspace(vec3(coord, scenedepth1));

            vec3 scenepos0  = view_scenespace(viewpos0);
            vec3 scenepos1  = view_scenespace(viewpos1);

            vec3 sdir       = normalize(scenepos1);

            float vdotl     = dot(normalize(viewpos1), lightvecView);

            bool water      = mat_id == 102;

            mat2x3 atmos;
                atmos[0] = vec3(0.0);
                atmos[1] = vec3(1.0);

            #ifdef vwater_enabled
                if (water || isEyeInWater == 1) compute_water(atmos, scenepos0, scenepos1, sdir, vdotl);
                #ifdef vfog_translucents
                else compute_fog2(atmos, scenepos0, scenepos1, sdir, !landMask(scenedepth1), vdotl);
                #endif
            #else
                if (!water) compute_fog2(atmos, scenepos0, scenepos1, sdir, !landMask(scenedepth1), vdotl);
            #endif

            tex5.rgb    = atmos[0];
            tex6.rgb    = atmos[1];
        }
    }
    #endif

    /*DRAWBUFFERS:056*/
    gl_FragData[0]  = make_drawbuffer(scenecol);
    gl_FragData[1]  = clamp_drawbuffer(tex5);
    gl_FragData[2]  = clamp_drawbuffer(tex6);
}