#version 400 compatibility

/*
====================================================================================================

    Copyright (C) 2020 RRe36

    All Rights Reserved unless otherwise explicitly stated.


    By downloading this you have agreed to the license and terms of use.
    These can be found inside the included license-file
    or here: https://rre36.github.io/license/

    Violating these terms may be penalized with actions according to the Digital Millennium
    Copyright Act (DMCA), the Information Society Directive and/or similar laws
    depending on your country.

====================================================================================================
*/

#include "/lib/common.glsl"
#include "/lib/util/encoders.glsl"

in vec2 coord;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D colortex5;
uniform sampler2D colortex6;

uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform int frameCounter;
uniform int isEyeInWater;

uniform float far;
uniform float near;

uniform vec2 pixelSize, viewSize;

uniform vec3 upvec;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;

vec3 decode3x8(float a){
    int bf = int(a*65535.);
    return vec3(bf%32, (bf>>5)%64, bf>>11) / vec3(31,63,31);
}

int decodemat_id16(float x) {
    return int(x*65535.0);
}

vec3 blend_translucencies(vec3 scenecol, vec4 translucents, vec3 albedo) {
    vec3 color  = scenecol;
        color  *= mix(vec3(1.0), albedo, translucents.a);
        color   = mix(color, translucents.rgb, translucents.a);

    return color;
}

vec3 apply_fogdata(vec3 scenecol, mat2x3 data) {
    return scenecol.rgb*data[1] + data[0];
}

float depth_lin(float depth) {
    return (2.0*near) / (far+near-depth * (far-near));
}

vec3 view_screenspace(vec3 viewpos) {
    return ((projMAD(gbufferProjection, viewpos) / -viewpos.z)) *0.5 + 0.5;
}

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

float screen_viewspace(float depth, mat4 projInv) {
    depth   = depth * 2.0 - 1.0;
    return projInv[3].z / (projInv[2].w * depth + projInv[3].w);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

#include "/lib/frag/bluenoise.glsl"

#ifdef reflections_enabled
//based on robobo1221's shaders because my old ssr is shit
vec4 ssrTrace(vec3 rdir, vec3 vpos, vec3 screenpos, float dither, float nDotV) {
    const int steps     = 16;
    const int refine    = 5;

    float rlength   = ((vpos.z + rdir.z * far * sqrt(3.0)) > -near) ? (-near - vpos.z) / rdir.z : far * sqrt(3.0);
    vec3 dir        = normalize(view_screenspace(rdir * rlength + vpos) - screenpos);
        dir.xy      = normalize(dir.xy);

    float maxlength = rcp(float(steps));
    float minlength = maxlength*0.05;

    float steplength = mix(minlength, maxlength, nDotV) * (dither);
    float stepweight = 1.0 / abs(dir.z);

    vec3 pos        = screenpos + dir*steplength;

    float depth     = texture(depthtex1, pos.xy).x;
    bool rayhit     = false;

    int i       = steps;
    int j       = refine;

    while (--i > 0) {
        steplength = clamp((depth - pos.z) * stepweight, minlength, maxlength);
        pos += dir*steplength;
        depth = texture(depthtex1, pos.xy).x;

        if (saturate(pos) != pos) return vec4(0.0);

        if (depth <= pos.z) break;
    }
    float mdepth    = depth;

    vec3 refpos     = pos;
    float refdepth  = depth;

    while (--j > 0) {
        refpos      = dir * clamp((depth - pos.z)*stepweight, -steplength, steplength) + pos;
        refdepth    = texture(depthtex1, refpos.xy).x;
        bool rayhit = refdepth <= refpos.z;

        pos         = rayhit ? refpos : pos;
        depth       = rayhit ? refdepth : depth;

        steplength *= 0.5;
    }

    float sdepth    = texture(depthtex1, pos.xy).x;

    if (sdepth >= 1.0) return vec4(0.0);

    bool visible    = abs(pos.z - mdepth) * min(stepweight, 400.0) <= maxlength;

    return visible ? vec4(texture(colortex0, pos.xy).rgb, 1.0) : vec4(0.0);
}

float fresnel_schlick(vec3 dir, vec3 normal) {
    vec3 halfdir    = normalize(normal + dir);

    float cosine    = dot(halfdir, dir);
    float fresnel   = max(cosine, 0.0);

    return pow3(fresnel);
}

#include "/lib/atmos/project.glsl"

vec2 unproject_sphere(vec3 dir) {
    vec2 lonlat     = vec2(atan(-dir.x, dir.z), acos(dir.y));
    return lonlat * vec2(rcp(tau), rcp(pi)) + vec2(0.5, 0.0);
}

void reflection(inout vec3 scenecol, in vec3 normal, in vec3 scenenormal, in float depth, vec2 lmap, int mat_id) {
    vec3 viewpos    = screen_viewspace(vec3(coord, depth));
    vec3 viewdir    = normalize(viewpos);
    vec3 scenepos   = view_scenespace(viewpos);
    vec3 scenedir   = normalize(scenepos);
    
    float nDotV     = saturate(dot(normal, viewdir));
    vec3 rdir       = normalize(reflect(viewdir, normal));

    float dither    = dither_bluenoise();

    #ifdef ssr_enabled
    vec4 ssr        = ssrTrace(rdir, viewpos, vec3(coord, depth), dither, nDotV);
    #else
    const vec4 ssr  = vec4(0.0);
    #endif

    float fresnel   = max(fresnel_schlick(viewdir, normal), 0.02);

    float fmult     = pow2(sstep(lmap.y, 0.15, 0.95));
        fmult      *= saturate(1.0 - dot(-upvec, scenenormal)) * 0.9 + 0.1;

    vec2 spherecoord = unproject_sphere(normalize(mat3(gbufferModelViewInverse)*rdir));
        spherecoord = spherecoord*rcp(SKYREF_LOD) + vec2(0.55);

    vec3 reflection = texture(colortex5, spherecoord).rgb*fmult;
        reflection  = mix(reflection, ssr.rgb, ssr.a);

        if (mat_id == 103) fresnel *= 0.35;

    scenecol        = mix(scenecol.rgb, reflection, fresnel*max(fmult, ssr.a));
}
#endif
/*
vec4 textureBilateral(sampler2D tex, sampler2D depth, const int lod, float fdepth, const vec2 offset) {
    vec4 data   = vec4(0.0);
    float sum   = 0.0;
    ivec2 posD  = ivec2(coord*viewSize);
    ivec2 posT  = ivec2((coord+offset)*viewSize*rcp(float(lod)));
    vec3 zmult  = vec3((far*near)*2.0, far+near, far-near);
        fdepth  = depth_lin(fdepth);
    
    for (int i = -1; i<2; i++) {
        for (int j = -1; j<2; j++) {
            ivec2 tcDepth = posD + ivec2(i, j)*lod;
            float dsample = depth_lin(texelFetch(depth, tcDepth, 0).x);
            float w     = abs(dsample-fdepth)*zmult.x<1.0 ? 1.0 : 1e-5;
            ivec2 ct    = posT + ivec2(i, j);
            data       += texelFetch(tex, ct, 0)*w;
            sum        += w;
        }
    }
    data *= rcp(sum);

    return data;
}*/
/*
mat2x3 textureBilateralDouble(sampler2D tex1, sampler2D tex2, sampler2D depth, const float lod, float fdepth, const vec2 offset) {
    mat2x3 data = mat2x3(0.0);
    float sum   = 0.0;
    ivec2 posD  = ivec2(coord*viewSize);
    ivec2 posT  = ivec2((coord*rcp(lod)+offset)*viewSize);
    vec3 zmult  = vec3((far*near)*2.0, far+near, far-near);
        fdepth  = depth_lin(fdepth);
    
    for (int i = -1; i<2; i++) {
        for (int j = -1; j<2; j++) {
            ivec2 tcDepth = posD + ivec2(vec2(i, j)*lod);
            float dsample = depth_lin(texelFetch(depth, tcDepth, 0).x);
            float w     = abs(dsample-fdepth)*zmult.x<1.0 ? 1.0 : 1e-5;
            ivec2 ct    = posT + ivec2(i, j);
            data[0]    += texelFetch(tex1, ct, 0).rgb*w;
            data[1]    += texelFetch(tex2, ct, 0).rgb*w;
            sum        += w;
        }
    }
    data *= rcp(max(sum, 1e-16));

    return data;
}
*/
const ivec2 kernelO_3x3[9]  = ivec2[9](
    ivec2(-1, -1), ivec2(0, -1), ivec2(1, -1),
    ivec2(-1, 0),  ivec2(0, 0),  ivec2(1, 0),
    ivec2(-1, 1),  ivec2(0, 1),  ivec2(1, 1)
);

mat2x3 bilateral_upscale(sampler2D tex1, sampler2D tex2, sampler2D depth, const float lod, float sdepth, const vec2 offset) {
    mat2x3 total = mat2x3(0.0);
    float sum   = 0.0;
    ivec2 depth_pos     = ivec2(coord * viewSize);
    ivec2 tex_pos       = ivec2((coord * rcp(lod) + offset) * viewSize);

        sdepth  = depth_lin(sdepth);

    for (int i = 0; i<9; i++) {
        ivec2 d  = kernelO_3x3[i];

        float blur  = expf(-dot(d, d) * 0.1);

        ivec2 dp    = depth_pos + ivec2(d * lod);
        float cdepth    = texelFetch(depth, dp, 0).x;
        float cdlin     = depth_lin(cdepth);
        float weight    = expf(-max(distance(sdepth, cdlin) * far - 0.15, 0.0) * 0.5);
        ivec2 tp    = tex_pos + d;
        total[0]   += texelFetch(tex1, tp, 0).rgb * weight;
        total[1]   += texelFetch(tex2, tp, 0).rgb * weight;
        sum        += weight;
    }

    return total * rcp(max(sum, 1e-20));
}

void main() {
    vec4 scenecol   = stex(colortex0);
    vec4 tex1       = stex(colortex1);
    vec4 tex2       = stex(colortex2);
    vec4 tex3       = stex(colortex3);

    vec2 scenelmap  = decode2x8(tex1.z);
    vec3 scenenormal = decodeNormal(tex1.xy);
    vec3 viewnormal = normalize(mat3(gbufferModelView) * scenenormal);

    vec3 translucent_albedo = pow2(decode3x8(tex2.g));

    float scenedepth0 = stex(depthtex0).r;
    float scenedepth1 = stex(depthtex1).r;

    int mat_id      = decodemat_id16(tex2.x);

    bool water      = mat_id == 102;

    #if (defined vwater_enabled || defined vfog_translucents)
        bool translucent = (scenedepth0<scenedepth1);

        mat2x3 atmos1;
            atmos1[0] = vec3(0.0);
            atmos1[1] = vec3(1.0);

        if (translucent || isEyeInWater == 1) {
            vec2 fcoord_atmos = saturate(coord*rcp(2.02) + vec2(0.0, 0.5));

            #ifdef vwater_bilateral
                atmos1      = bilateral_upscale(colortex5, colortex6, depthtex0, 2.02, scenedepth0, vec2(0.0, 0.5));
            #else
                atmos1[0]   = texture(colortex5, fcoord_atmos).rgb;
                atmos1[1]   = texture(colortex6, fcoord_atmos).rgb;
            #endif
        }
        
        if (translucent && isEyeInWater == 0) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos1);
    #endif

    #ifdef vfog_enabled
        vec2 fcoord_atmos = saturate(coord*0.5 + vec2(0.5, 0.0));

        mat2x3 atmos;
            atmos[0] = vec3(0.0);
            atmos[1] = vec3(1.0);

        #ifdef vfog_bilateral
            atmos       = bilateral_upscale(colortex5, colortex6, depthtex0, 2, scenedepth0, vec2(0.5, 0.0));
        #else
            atmos[0]      = texture(colortex5, fcoord_atmos).rgb;
            atmos[1]      = texture(colortex6, fcoord_atmos).rgb;
        #endif

        if (isEyeInWater == 1) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos);
    #endif

    scenecol.rgb    = blend_translucencies(scenecol.rgb, tex3, translucent_albedo);

    #ifdef reflections_enabled
        if (water || mat_id == 103) reflection(scenecol.rgb, viewnormal, scenenormal, scenedepth0, scenelmap, mat_id);
    #endif

    #ifdef vfog_enabled
        if (isEyeInWater == 0) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos);
    #endif
    
    #if (defined vwater_enabled || defined vfog_translucents)
        if (isEyeInWater == 1) scenecol.rgb = apply_fogdata(scenecol.rgb, atmos1);
    #endif

    if (mat_id == 3) {
        //if (landMask(scenedepth0)) scenecol.rgb = scenecol.rgb*0.6 + vec3(0.5)*v3avg(scenecol.rgb);
        scenecol.rgb = scenecol.rgb*0.7 + vec3(0.8)*v3avg(scenecol.rgb);
    }

    //if (isnan4(scenecol) || isinf4(scenecol)) scenecol.rgb = vec3(0.0, 40.0, 0.0);

    /*DRAWBUFFERS:05*/
    gl_FragData[0]  = make_drawbuffer(scenecol);
    gl_FragData[1]  = vec4(0.0, 0.0, 0.0, 1.0);
}