#version 400 compatibility

/*
====================================================================================================

    Copyright (C) 2020 RRe36

    All Rights Reserved unless otherwise explicitly stated.


    By downloading this you have agreed to the license and terms of use.
    These can be found inside the included license-file
    or here: https://rre36.github.io/license/

    Violating these terms may be penalized with actions according to the Digital Millennium
    Copyright Act (DMCA), the Information Society Directive and/or similar laws
    depending on your country.

====================================================================================================
*/

#include "/lib/common.glsl"

const int noiseTextureResolution = 256;

in vec2 coord;

flat in vec3 cloudSunlight;

flat in mat4x3 light_color;
//flat in mat3x3 sky_color;

uniform sampler2D colortex0;
uniform sampler2D colortex5;

uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform float aspectRatio;
uniform float eyeAltitude;
uniform float frameTimeCounter;
uniform float far, near;
uniform float wetness;

uniform int frameCounter;
uniform int worldTime;

uniform vec2 viewSize;

uniform vec3 upvec, upvecView;
uniform vec3 sunvec, sunvecView;
uniform vec3 moonvec, moonvecView;
uniform vec3 lightvec;
uniform vec3 cloud_lvec, cloud_lvecView;

uniform vec3 cameraPosition;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;

#include "/lib/atmos/aconst.glsl"

vec3 screen_viewspace(vec3 screenpos, mat4 projInv) {
    screenpos   = screenpos*2.0-1.0;

    vec3 viewpos    = vec3(vec2(projInv[0].x, projInv[1].y)*screenpos.xy + projInv[3].xy, projInv[3].z);
        viewpos    /= projInv[2].w*screenpos.z + projInv[3].w;
    
    return viewpos;
}

vec3 screen_viewspace(vec3 screenpos) {
    return screen_viewspace(screenpos, gbufferProjectionInverse);
}

vec3 view_scenespace(vec3 viewpos, mat4 mvInv) {
    return viewMAD(mvInv, viewpos);
}

vec3 view_scenespace(vec3 viewpos) {
    return view_scenespace(viewpos, gbufferModelViewInverse);
}

float depth_lin(float depth) {
    return (2.0*near) / (far+near-depth * (far-near));
}

float get_mie(float x, float g) {
    float temp  = 1.0 + pow2(g) - 2.0*g*x;
    return (1.0 - pow2(g)) / ((4.0*pi) * temp*(temp*0.5+0.5));
}
/*
vec3 get_sky(vec3 viewvec) {
    vec3 v      = -viewvec;
    vec3 hvt    = normalize(-upvecView+v);
    vec3 hvb    = normalize(upvecView+v);
    vec3 sv     = normalize(sunvecView+v);
    vec3 mv     = normalize(moonvecView+v);
    float vdots = dot(viewvec, sunvecView);

    float hor_t = dot(hvt, v);
    float hor_b = dot(hvb, v);
    float sun   = 1.0-dot(sv, v);
    float hor   = lin_step(1.0-max(hor_t, hor_b * 0.93), 0.0, 0.31);

    float horizon   = pow5(hor);

    float hmie  = get_mie(vdots, 0.5)*rcp(0.5);
        horizon = mix(hmie * horizon, horizon * 0.95 + hmie * 0.05, (daytime.y + daytime.w) * 0.85 + 0.15);

    float zenith    = 1.0-lin_step(1.0-hor_t, 0.0, 0.2958);
        zenith      = exp(-zenith*1.8);

    float ld        = lin_step(min(hor_t, hor_b), 0.64, 0.71);
        ld          = (ld);

    float sunscatter = get_mie(vdots, 0.82)*rcp(2.0)*(1.0-daytime.w);
    float moonscatter = get_mie(dot(viewvec, moonvecView), 0.83)*rcp(2.0)*sqrt(daytime.w);

    vec3 sky    = sky_color[0]*zenith;
        sky     = mix(sky, sky_color[1], horizon);
        sky     = mix(sky, sky_color[2], hmie * horizon * (daytime.y));
        //sky     = mix(sky, sky_color[1], ld);
        sky    += sky_color[2]*sunscatter * (1.0 - pow2(daytime.w));
        sky    += light_color[3] * moonscatter;
        //sky     = hor > 0.99 ? vec3(horizon) : vec3( 0.0);

    return sky;
}*/

#include "/lib/atmos/project.glsl"

vec4 textureBicubic(sampler2D sampler, vec2 coord) {
	vec2 res = textureSize(sampler, 0);

	coord = coord * res - 0.5;

	vec2 f = fract(coord);
	coord -= f;

	vec2 ff = f * f;
	vec4 w0;
	vec4 w1;
	w0.xz = 1 - f; w0.xz *= w0.xz * w0.xz;
	w1.yw = ff * f;
	w1.xz = 3 * w1.yw + 4 - 6 * ff;
	w0.yw = 6 - w1.xz - w1.yw - w0.xz;

	vec4 s = w0 + w1;
	vec4 c = coord.xxyy + vec2(-0.5, 1.5).xyxy + w1 / s;
	c /= res.xxyy;

	vec2 m = s.xz / (s.xz + s.yw);
	return mix(
		mix(texture(sampler, c.yw), texture(sampler, c.xw), m.x),
		mix(texture(sampler, c.yz), texture(sampler, c.xz), m.x),
		m.y);
}

vec3 get_sun(vec3 viewvec) {
    vec3 v      = -viewvec;
    vec3 sv     = normalize(sunvecView+v);
    float sun   = dot(sv, v);

    float s   = 1.0-lin_step(sun, 0.0225, 0.0229);
        //s    *= 1.0-sstep(sun, 0.004, 0.0059)*0.5;

    return s*light_color[0]*2e2;
}

vec3 get_moon(vec3 viewvec, vec3 albedo) {
    vec3 v      = -viewvec;
    vec3 sv     = normalize(moonvecView+v);
    float sun   = dot(sv, v);

    float s   = 1.0-lin_step(sun, 0.03, 0.08);

    return albedo*s*light_color[3]*70.0;
}

#include "/lib/frag/noise.glsl"

vec2 rotate_coord(vec2 pos, const float angle) {
    return vec2(cos(angle)*pos.x + sin(angle)*pos.y, 
                cos(angle)*pos.y - sin(angle)*pos.x);
}

vec3 get_stars(vec3 spos, vec3 svec) {
    vec3 plane  = svec/(svec.y+length(svec.xz)*0.66);
    float rot   = worldTime*rcp(2400.0);
    plane.x    += rot*0.6;
    plane.yz    = rotate_coord(plane.yz, (25.0/180.0)*pi);
    plane      *= 1.25;
    vec2 uv1    = floor((plane.xz)*768)/768;
    vec2 uv2    = (plane.xz)*0.04;

    vec3 starcol = vec3(0.5, 0.68, 1.0);
        starcol  = mix(starcol, vec3(1.0, 0.7, 0.6), noise_2d(uv2).x * 0.7);
        starcol  = normalize(starcol)*(noise_2d(uv2*1.5).x+1.0);

    float star  = 1.0;
        star   *= noise_2d(uv1).x;
        star   *= noise_2d(uv1+0.1).x;
        star   *= noise_2d(uv1+0.26).x;

    star        = max(star-0.25, 0.0);
    star        = saturate(star*4.0);

    return star*starcol*0.07*sqrt(daytime.w);
}

/* ------ clouds ------ */

float fbm(vec3 pos, vec3 offset, const float persistence, const float scale, const int octaves) {
    float n     = 0.0;
    float a     = 1.0;
    vec3 shift  = offset;

    for (int i = 0; i<octaves; ++i) {
        n      += value_3d(pos + shift)*a;
        pos.xz  = rotate_coord(pos.xz, pi*0.33);
        pos    *= scale;
        a      *= persistence;
    }

    return n;
}


const float pc_size     = 0.00026;
float vcloud_tick       = frameTimeCounter*0.1;

const float pc_highedge     = pcloud_alt + pcloud_depth;
const float pc_midalt       = pcloud_alt + pcloud_depth * 0.5;

float cloud_mie_dumb(float cos_theta, float g) {
    float sqG   = pow2(g);
    float a     = (1.0-sqG) / (2.0 + sqG);
    float b     = (1.0 + pow2(cos_theta)) / (1.0 + sqG - 2.0*g*cos_theta);

    return max(1.5 * a*b + g*cos_theta, 0.0)*rcp(pi);
}
float cloud_mie(float x, float g) {
    float mie   = 1.0 + pow2(g) - 2.0*g*x;
        mie     = (1.0 - pow2(g)) / ((4.0*pi) * mie*(mie*0.5+0.5));
    return mie;
}

float cloud_phase_cirrus(float cos_theta, float g) {
    float a     = cloud_mie_dumb(cos_theta, 0.89*g);
    float b     = cloud_mie(cos_theta, -0.25*g) * 1.3;

    return mix(a, b, 0.3);
}

float pc_shape(in vec3 pos) {
    vec3 pos0 = pos/250.0;
    //pos0.xz    /= 1.0+max(vdist/10000.0, 0.0); 

    float tick  = frameTimeCounter*0.4;

    pos0    += (value_3d(pos0*2.0 + vec3(0.0, tick*0.01, 0.0))*2.0-1.0)*0.1;
    pos0.x  -= (value_3d(pos0*0.125 + vec3(0.0, tick*0.01, 0.0))*2.0-1.0)*1.2;

    pos0.x  *= 0.25;
    pos0.x  -= tick*0.01;

    //vec3 pos1   = pos0*vec3(1.0, 0.5, 1.0)+vec3(0.0, tick*0.01, 0.0);

    float noise = value_3d(pos0*vec3(1.0, 0.5, 1.0) + vec3(0.0, tick*0.01, 0.0));
            pos0 *= 2.0; pos0.x -= tick*0.017; pos0.z += noise*1.35; pos0.x += noise*0.5;

        noise  += (2.0-abs(value_3d(pos0)*2.0))*0.35;
            pos0 *= 3.0; pos0.xz -= tick*0.005; pos0.z += noise*1.35; pos0.x += noise*0.5; pos0.x *= 3.0; pos0.z *= 0.55;
            pos0.z  -= (value_3d(pos0*0.25+vec3(0.0, tick*0.01, 0.0))*2.0-1.0)*0.4;

        noise  += (3.0-abs(value_3d(pos0)*3.0))*0.035;
            pos0 *= 3.0; pos0.z *= 0.5; pos0.xz -= tick*0.005;

        noise  += (3.0-abs(value_3d(pos0)*3.0))*0.035;
            pos0 *= 3.0; pos0.xz -= tick*0.005;

        noise  += (3.0-abs(value_3d(pos0)*3.0))*0.025;
            pos0 *= 4.0; pos0.xz -= tick*0.005;

        noise  += value_3d(pos0)*0.054;
            //pos0 *= 3.0;

        //noise  += value_3d(pos0)*0.024;

        noise  /= 1.575;
        
    float shape = noise;

    float altWeight = 1.0 - (lin_step(distance(pos.y, pc_midalt), 15.0, (pcloud_depth*0.5)));
        altWeight   = sqrt(sqrt(altWeight));

    float coverage    = mix(0.4, 0.8, wetness);
        shape       = max(shape-(1.0-coverage*altWeight), 0.0)/coverage;
        shape       = cube_smooth(shape);

        shape *= sqrt(altWeight);

    return max(shape, 0.0);
}

float pc_directOD(in vec3 pos, const int steps) {
    vec3 dir    = cloud_lvec;

    float stepsize  = (pcloud_depth / steps);

    float od = 0.0;
    for(int i = 0; i < steps; ++i, pos += dir * stepsize) {
        if(pos.y > pc_highedge || pos.y < pcloud_alt) continue;

        float attenCoeff = pc_shape(pos);
        od += attenCoeff*stepsize;
    } 

    return od;
}
float pc_skyOD(in vec3 pos, const int steps) {
    vec3 dir    = vec3(0.0, 1.0, 0.0);

    float stepsize = (pcloud_depth / steps);
        stepsize  *= (1.0-lin_step(pos.y, pcloud_alt, pc_highedge))*0.9+0.1;

    float od = 0.0;
    for(int i = 0; i < steps; ++i, pos += dir * stepsize) {
        if(pos.y > pc_highedge || pos.y < pcloud_alt) break;
        float attenCoeff = pc_shape(pos);
        od += attenCoeff*stepsize;
    } 

    return od;
}

void compute_pc(inout vec3 scenecol, in vec3 vvec, in vec3 wvec, in float layer_transmittance) {
    //float within    = sstep(eyeAltitude, vcloud_alt-100.0, vcloud_alt) * (1.0-sstep(eyeAltitude, c_highedge, vc_highedge+100.0));
    bool visible    = wvec.y>0.0;

    if (visible) {
        //const float pc_midalt = pcloud_alt+pcloud_depth*0.5;

        bool is_below   = eyeAltitude<pc_midalt;

        vec3 s      = wvec*((pc_midalt-eyeAltitude)/wvec.y);

        if (wvec.y<0.0 && is_below || wvec.y>0.0 && !is_below) s = vec3(0.0);

        vec3 rpos    = s + cameraPosition;

        //float dither = dither_bluenoise();

        float rlength   = pcloud_depth*0.5;

        vec3 scatter    = vec3(0.0);
        float transmittance = 1.0;
        float fade      = 0.0;
        float fdist     = pcloud_clip+1.0;

        vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloudSunlight : light_color[3];
        vec3 skylight   = light_color[1]*0.2;

        float vdotl     = dot(vvec, cloud_lvecView);

        float pfade     = saturate(cloud_mie(vdotl, 0.65));

        const float sigma_a = 1.0;         //absorption coeff
        const float sigma_s = 0.06;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
        const float sigma_t = 0.06;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
/*    
        const float sigma_a = 1.00;         //absorption coeff
        const float sigma_s = 0.035;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
        const float sigma_t = 0.04;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus
*/
        /* --- calculate clouds --- */

        float dist  = distance(rpos, cameraPosition);
        float dfade = saturate(dist/pcloud_clip);
        if (!((1.0-dfade)<0.01)) {

        rpos.xz    *= 1.0-sstep(distance(rpos.xz, cameraPosition.xz), 0.0, 80000.0)*0.5;

        float density   = pc_shape(rpos);
        if (!(density<=0.0)) {

        if (fdist>pcloud_clip) fdist = dist;

        fade    = lin_step(dfade, 0.85, 0.99);

        float extinction = density * sigma_t;
        float stept     = expf(-extinction*rlength);
        float integral  = (1.0 - stept) / sigma_t;

        vec3 result_s   = vec3(0.0);

        float directod  = pc_directOD(rpos, 4)*sigma_a;
        float skyod     = pc_skyOD(rpos, 4)*sigma_a;

        float powder    = 1.0 - expf(-density * 3.0);
        float dpowder   = mix(powder, 1.0, pfade);

        //float phase = cloud_phase_cirrus(vdotl, 1.0) * 1.25;
        //float phase2 = cloud_phase_cirrus(vdotl, 0.2);

        //result_s.x += max(expf(-directod * sigma_t) * phase, expf(-directod * sigma_t * 0.2) * 0.75 * phase2) * dpowder * sigma_s;
        //result_s.y += max(expf(-skyod * sigma_t), expf(-skyod * sigma_t * 0.2) * 0.75) * powder * sigma_s;

        
        for (int j = 0; j<5; ++j) {
            float n     = float(j);

            float s_d   = sigma_s * pow(0.5, n);    //scatter derivate
            float t_d   = sigma_t * pow(0.5, n);    //transmittance/attentuation derivate
            float phase = cloud_phase_cirrus(vdotl, pow(0.5, n));  //phase derivate

            result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
            result_s.y += expf(-skyod*t_d) * powder * s_d;
        }

        scatter    += result_s * integral;

        transmittance *= stept;
        }
        }

        fdist   = max(fdist, 0.0);

        scatter.x *= pi * 1.5;

        vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

        fade        = saturate(1.0-fade);

        const vec3 extinct_coeff = vec3(3e-4);

        vec3 atmosfade  = expf(-extinct_coeff*vec3(fdist));

        float skyfade   = expf(-fdist*cloud_atmos_density)*fade;

        scenecol        = mix(scenecol, scenecol * transmittance + color, atmosfade*skyfade);
    }
}

/* ------ ambient occlusion and gi ------ */

#include "/lib/light/ao.glsl"

/* ------ sky reflection ------ */

vec2 sincos(float x) {
    return vec2(sin(x), cos(x));
}

vec3 project_sphere(vec2 coord) {
    coord  *= vec2(tau, pi);
    vec2 lon = sincos(coord.x) * sin(coord.y);
    return vec3(lon.x, cos(coord.y), lon.y);
}

#include "/lib/frag/bluenoise.glsl"

#include "/lib/atmos/vcloud.glsl"

void compute_vcr(inout vec3 scenecol, vec3 wvec, inout float layer_transmittance) {
    bool visible    = wvec.y>0.0;
    const float eyeAltitude = 64.0;
    const int bstep     = 25;
    const float clip    = 6000.0;

    if (visible) {
        vec3 bs     = wvec*((vcloud_alt-eyeAltitude)/wvec.y);
        vec3 ts     = wvec*((vcloud_maxalt-eyeAltitude)/wvec.y);

        vec3 spos   = bs;
        vec3 epos   = ts;

        float dither = dither_bluenoise_s();

        const float bl  = vcloud_depth/bstep;
        float stepl     = length((epos-spos)/bstep);
        float stepcoeff = stepl/bl;
            stepcoeff   = 0.45+clamp(stepcoeff-1.1, 0.0, 3.0)*0.5;
        uint steps      = uint(bstep*stepcoeff);

        vec3 rstep  = (epos-spos)/steps;
        vec3 rpos   = rstep*dither + spos + vec3(cameraPosition.x, eyeAltitude, cameraPosition.z);
        float rlength = length(rstep);

        vec3 scatter    = vec3(0.0);
        float transmittance = 1.0;
        float fade      = 0.0;
        float fdist     = clip + 1.0;

        vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloudSunlight : light_color[3];
        vec3 skylight   = light_color[1]*0.2;

        float vdotl     = dot(wvec, cloud_lvec);

        float pfade     = saturate(cloud_mie(vdotl, 0.65));

        const float sigma_a = 1.00;         //absorption coeff
        const float sigma_s = 0.40;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
        const float sigma_t = 0.40;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus

        for (uint i = 0; i<steps; ++i, rpos += rstep) {
            if (transmittance < 0.1) break;
            if (rpos.y < vcloud_alt || rpos.y > vcloud_maxalt) continue;
            float dist  = distance(rpos.xz, cameraPosition.xz);
            if (dist > clip) continue;
            float dfade = saturate(dist/clip);
            //if ((1.0-dfade)<0.01) continue;

            float density   = vcloud_shape(rpos);
            if (density<=0.0) continue;
            
            float f     = lin_step(dfade, 0.75, 0.99);

            if (fdist>clip) {
                fdist   = dist;
                fade    = f;
            } else {
                fdist   = mix(fdist, dist, transmittance);
                fade    = mix(fade, f, transmittance);
            }

            float extinction = density * sigma_t;
            float stept     = expf(-extinction*rlength);
            float integral  = (1.0 - stept) / sigma_t;

            vec3 result_s   = vec3(0.0);

            float directod  = vc_directOD(rpos, 4)*sigma_a;
            float skyod     = vc_skyOD(rpos, 3)*sigma_a;

            float powder    = 1.0 - expf(-density * vc_powder_K);
            float dpowder   = min(powder, 1.0-expf(-directod*2.0));
                dpowder     = mix(dpowder, 1.0, pfade);

            float s_d   = sigma_s * 2.0;
            float t_d   = sigma_t * 2.0;

            for (uint j = 0; j<6; ++j) {
                float n     = float(j);

                s_d *= 0.5;
                t_d *= 0.5;
                float phase = cloud_phase(vdotl, pow(0.5, n));  //phase derivate

                result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
                result_s.y += expf(-skyod*t_d) * powder * s_d;
            }
            //result_s.z = vc_bouncelight(rpos) * (powder * 0.5 + 0.5);

            scatter    += result_s * integral * transmittance;

            transmittance *= stept;
        }
        transmittance = lin_step(transmittance, 0.1, 1.0);
        scatter.x *= 2.4;

        vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

        fade        = saturate(1.0-fade);

        const vec3 extinct_coeff = vec3(3e-4);

        vec3 atmosfade  = expf(-extinct_coeff * vec3(fdist));
        float skyfade   = expf(-fdist * cloud_atmos_density);

        layer_transmittance *= transmittance;

        scenecol = mix(scenecol, scenecol * transmittance + color, atmosfade * skyfade * fade);
    }
}

void compute_pcr(inout vec3 scenecol, vec3 wvec, inout float layer_transmittance) {
    bool visible    = wvec.y>0.0;
    const float eyeAltitude = 64.0;

    if (visible) {
        //const float pc_midalt = pcloud_alt+pcloud_depth*0.5;

        bool is_below   = eyeAltitude<pc_midalt;

        vec3 s      = wvec*((pc_midalt-eyeAltitude)/wvec.y);

        vec3 rpos    = s + vec3(cameraPosition.x, eyeAltitude, cameraPosition.z);

        //float dither = dither_bluenoise();

        float rlength   = pcloud_depth*0.5;

        vec3 scatter    = vec3(0.0);
        float transmittance = 1.0;
        float fade      = 0.0;
        float fdist     = pcloud_clip+1.0;

        vec3 sunlight   = (worldTime>23000 || worldTime<12900) ? cloudSunlight : light_color[3];
        vec3 skylight   = light_color[1]*0.2;

        float vdotl     = dot(wvec, cloud_lvec);

        float pfade     = saturate(cloud_mie(vdotl, 0.65));

        const float sigma_a = 1.0;         //absorption coeff
        const float sigma_s = 0.06;         //scattering coeff, can technically be assumed to be sigma_t since the albedo is close to 1.0
        const float sigma_t = 0.06;         //extinction coeff, 0.05-0.12 for cumulus, 0.04-0.06 for stratus

        /* --- calculate clouds --- */

        float dist  = distance(rpos, cameraPosition);
        float dfade = saturate(dist/pcloud_clip);
        if (!((1.0-dfade)<0.01)) {

        rpos.xz    *= 1.0-sstep(distance(rpos.xz, cameraPosition.xz), 0.0, 80000.0)*0.5;

        float density   = pc_shape(rpos);
        if (density>0.0) {

        if (fdist>pcloud_clip) fdist = dist;

        fade    = lin_step(dfade, 0.85, 0.99);

        float extinction = density * sigma_t;
        float stept     = expf(-extinction*rlength);
        float integral  = (1.0 - stept) / sigma_t;

        vec3 result_s   = vec3(0.0);

        float directod  = pc_directOD(rpos, 4)*sigma_a;
        float skyod     = pc_skyOD(rpos, 3)*sigma_a;

        float powder    = 1.0 - expf(-density * 3.0);
        float dpowder   = mix(powder, 1.0, pfade);
        
        for (int j = 0; j<5; ++j) {
            float n     = float(j);

            float s_d   = sigma_s * pow(0.5, n);    //scatter derivate
            float t_d   = sigma_t * pow(0.5, n);    //transmittance/attentuation derivate
            float phase = cloud_phase_cirrus(vdotl, pow(0.5, n));  //phase derivate

            result_s.x += expf(-directod*t_d) * phase * dpowder * s_d;
            result_s.y += expf(-skyod*t_d) * powder * s_d;
        }

        scatter    += result_s * integral;

        transmittance *= stept;
        }
        }

        fdist   = max(fdist, 0.0);

        scatter.x *= pi * 1.5;

        vec3 color  = (sunlight*scatter.x) + (skylight*scatter.y);

        fade        = saturate(1.0-fade);

        const vec3 extinct_coeff = vec3(3e-4);

        vec3 atmosfade  = expf(-extinct_coeff*vec3(fdist));

        float skyfade   = expf(-fdist*cloud_atmos_density)*fade;

        scenecol        = mix(scenecol, scenecol * transmittance + color, atmosfade*skyfade * layer_transmittance);
    }
}

void main() {
    vec4 scenecol   = stex(colortex0);
    float scenedepth = stex(depthtex1).x;

    vec3 viewpos    = screen_viewspace(vec3(coord, scenedepth));
    vec3 viewvec    = normalize(viewpos);
    vec3 scenepos   = view_scenespace(viewpos);
    vec3 svec       = normalize(scenepos);

    if (!landMask(scenedepth)) {
        //vec3 skycol = get_sky(viewvec);
        //vec3 skycol = getAtmosphericScattering(svec, sunvec, light_color[0]);
        #ifdef sky_hq_lookup
            vec3 skycol   = textureBicubic(colortex5, projectSky(svec)).rgb;
        #else
            vec3 skycol   = texture(colortex5, projectSky(svec)).rgb;
        #endif

        vec3 sun    = get_sun(viewvec);
        vec3 moonstars = get_moon(viewvec, scenecol.rgb) + get_stars(scenepos, svec);

        vec3 air_transmittance = textureBicubic(colortex5, projectSky(svec)+vec2(0.0, 0.3)).rgb;

        scenecol.rgb = skycol + (sun+moonstars)*air_transmittance;

        //scenecol.rgb = skycol + (sun+moonstars)*sstep(svec.y, -0.03, 0.02);

        #ifdef pcloud_enabled
            compute_pc(scenecol.rgb, viewvec, svec, 1.0);
        #endif
    }

    vec4 return3    = vec4(0.0);

    #ifdef ambientOcclusion_enabled
        vec2 ao_coord   = (coord)*2.0;

        if (clamp(ao_coord, -0.003, 1.003) == ao_coord) {
            vec2 coord  = ao_coord;
            float scenedepth = stex(depthtex1).x;
            float ao    = calculate_dbao(depthtex1, scenedepth, coord);
            return3.x   = ao;
        }
    #endif

    /* ------ sky reflection ------ */

    vec4 return5    = stex(colortex5);
    vec2 skyref_coord = (coord-vec2(0.55))*SKYREF_LOD;

    if (clamp(skyref_coord, -0.003, 1.003) == skyref_coord) {
        vec2 coord  = saturate(skyref_coord);
        vec3 svec   = project_sphere(coord)*vec3(1.0, 1.0, -1.0);
        return5.rgb = texture(colortex5, projectSky(svec)).rgb;
        
        #ifdef cloud_reflections_enabled
            float layer_transmittance = 1.0;

            #ifdef vcloud_enabled
                compute_vcr(return5.rgb, svec, layer_transmittance);
            #endif

            #ifdef pcloud_enabled
                compute_pcr(return5.rgb, svec, layer_transmittance);
            #endif
        #endif
    }

    /*DRAWBUFFERS:035*/
    gl_FragData[0]  = make_drawbuffer(scenecol);
    gl_FragData[1]  = clamp_drawbuffer(return3);
    gl_FragData[2]  = clamp_drawbuffer(return5);
}