var ASM = Java.type('net.minecraftforge.coremod.api.ASMAPI');
var Opcodes = Java.type('org.objectweb.asm.Opcodes');
var InsnList = Java.type('org.objectweb.asm.tree.InsnList');
var InsnNode = Java.type('org.objectweb.asm.tree.InsnNode');
var JumpInsnNode = Java.type('org.objectweb.asm.tree.JumpInsnNode');
var LdcInsnNode = Java.type('org.objectweb.asm.tree.LdcInsnNode');
var MethodInsnNode = Java.type('org.objectweb.asm.tree.MethodInsnNode');
var FieldInsnNode = Java.type('org.objectweb.asm.tree.FieldInsnNode');
var TypeInsnNode = Java.type('org.objectweb.asm.tree.TypeInsnNode');
var VarInsnNode = Java.type('org.objectweb.asm.tree.VarInsnNode');
var LabelNode = Java.type('org.objectweb.asm.tree.LabelNode');


function insnLoadThis(comment) {
	return insnLoadObj(0, comment);
}

function insnLoadObj(varIndex, comment) {
	return new VarInsnNode(Opcodes.ALOAD, varIndex);
}

function insnSaveObj(varIndex, comment) {
	return new VarInsnNode(Opcodes.ASTORE, varIndex);
}

function insnLoadInt(varIndex, comment) {
	return new VarInsnNode(Opcodes.ILOAD, varIndex);
}

function insnSaveInt(varIndex, comment) {
	return new VarInsnNode(Opcodes.ISTORE, varIndex);
}

function insnDup(comment) {
	return new InsnNode(Opcodes.DUP);
}

function insnDup2(comment) {
	return new InsnNode(Opcodes.DUP2);
}

function insnPop(comment) {
	return new InsnNode(Opcodes.POP);
}

function insnPushValue(value, comment) {
	return new LdcInsnNode(value)
}

function insnSwap(comment) {
	return new InsnNode(Opcodes.SWAP);
}

function insnPushInt(value, comment) {
	if (value >= 0 && value <= 5) {
		var opcode = Opcodes.ICONST_0 + value;
		return new InsnNode(opcode);
	} else
		return new LdcInsnNode(value);
}

function insnCheckCast(desc) {
	return new TypeInsnNode(Opcodes.CHECKCAST, desc)
}

function insnIfLE_JumpTo(label) {
	return new JumpInsnNode(Opcodes.IFLE, label);
}

function insnJumpTo(label) {
	return new JumpInsnNode(Opcodes.GOTO, label)
}

function insnReturnInt() {
	return new InsnNode(Opcodes.IRETURN);
}

function insnReturn() {
	return new InsnNode(Opcodes.RETURN);
}

function insnCallHandler(handler) {
	return ASM.buildMethodCall(handler.class, handler.methodName, handler.methodDesc, ASM.MethodType.STATIC);
}

function insnInvokeVirtual(method) {
	return ASM.buildMethodCall(method.class, method.methodName, method.methodDesc, ASM.MethodType.VIRTUAL);
}

function insnInvokeInterface(method) {
	return ASM.buildMethodCall(method.class, method.methodName, method.methodDesc, ASM.MethodType.INTERFACE);
}

function insnGetField(field) {
	return new FieldInsnNode(Opcodes.GETFIELD, field.class, field.fieldName, field.fieldDesc);
}

function insnGetStatic(field) {
	return new FieldInsnNode(Opcodes.GETSTATIC, field.class, field.fieldName, field.fieldDesc);
}

function instructionsDupValue2(comment)
{
	return [new InsnNode(Opcodes.DUP2), new InsnNode(Opcodes.POP)];
}

function instructionsDupValue3(comment)
{
	return [new InsnNode(Opcodes.DUP2_X1), new InsnNode(Opcodes.POP2), new InsnNode(Opcodes.DUP_X2)]
}

function instructionsIfTrueReturn() {
	var label = new LabelNode();
	return [new JumpInsnNode(Opcodes.IFEQ, label), insnReturn(), label];
}

function listOf(instructions) {
	var list = new InsnList();    
	for (var i = 0; i < instructions.length; i++) {
		list.add(instructions[i]);
	}
	return list;
}

function insertInstructionsBefore(method, node, instructions) {
	return insertInstructions(method, node, instructions, true);
}

function insertInstructionsAfter(method, node, instructions) {
	return insertInstructions(method, node, instructions, false);
}

function insertInstructions(method, node, instructions, before) {
	var list = listOf(instructions);
	var iterator = method.instructions.iterator();
    while (iterator.hasNext()) {
        var instruct = iterator.next();
		if (instruct instanceof MethodInsnNode) {
			logDebug(instruct.getOpcode() + "->" + instruct.owner + "." + instruct.name	+ instruct.desc);
            if (instruct.owner.equals(node.class) && instruct.name.equals(node.methodName) && instruct.desc.equals(node.methodDesc)) {
                if (before === true)
                    method.instructions.insertBefore(instruct, list);
                else
                    method.instructions.insert(instruct, list);
                logSuccess();
                return true;
            } 
		}	
	}
	logError("FAILED: target node not found!");
	return false;
}

function insertInstructionsBeforeAll(method, instructions) {
	var list = listOf(instructions);
	method.instructions.insert(list);
	logSuccess();
}

function newLabel() {
	return new LabelNode();
}

function resolveName(srgName) {
	return ASM.mapMethod(srgName);
}

function logSuccess() {
	logDebug("SUCCEEDED!");
}

function logDebug(message) {
	ASM.log("DEBUG", message);
}

function logError(message) {
	ASM.log("ERROR", message);
}
